# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # For this phase, we will:
        # 1. Move the robot to each location.
        # 2. Attempt to interact with objects (pick, pull, etc.) using available skills.
        # 3. Observe if any skill fails or if the environment state does not change as expected.
        # 4. Log the results to help identify the missing predicate.

        # Get all objects and locations from the positions dictionary
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []
        for name, pos in positions.items():
            if "drawer" in name:
                drawer_names.append(name)
            elif "handle" in name:
                handle_names.append(name)
            elif "loc" in name or "location" in name or "room" in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # If location_names is empty, try to infer from positions
        if not location_names:
            # Assume any object with 'pos' in its name is a location
            for name in positions:
                if "pos" in name or "location" in name or "room" in name:
                    location_names.append(name)

        # If still empty, just use all unique positions as locations
        if not location_names:
            location_names = list(set([tuple(pos) for pos in positions.values()]))

        # For demonstration, pick the first location as the starting point
        if len(location_names) == 0:
            print("[Exploration] No locations found in object_positions.")
            return

        # Try to move to each location and interact with objects
        for loc in location_names:
            try:
                # Move to location (if not already there)
                # Use execute_go if available
                # For execute_go, we need from and to locations
                # We'll assume the robot starts at the first location
                current_loc = location_names[0]
                if loc != current_loc:
                    print(f"[Exploration] Moving from {current_loc} to {loc}")
                    try:
                        obs, reward, done = execute_go(env, task, current_loc, loc)
                        current_loc = loc
                    except Exception as e:
                        print(f"[Exploration] execute_go failed: {e}")
                        continue

                # Try to pick up each object at this location
                for obj in object_names:
                    try:
                        print(f"[Exploration] Attempting to pick {obj} at {loc}")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Successfully picked {obj} at {loc}")
                        else:
                            print(f"[Exploration] Could not pick {obj} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

                # Try to pull each drawer handle at this location
                for drawer in drawer_names:
                    for handle in handle_names:
                        try:
                            print(f"[Exploration] Attempting to pull {drawer} with {handle} at {loc}")
                            obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                            if done:
                                print(f"[Exploration] Successfully pulled {drawer} with {handle} at {loc}")
                            else:
                                print(f"[Exploration] Could not pull {drawer} with {handle} at {loc}")
                        except Exception as e:
                            print(f"[Exploration] execute_pull failed for {drawer} with {handle} at {loc}: {e}")

                # Try to push each drawer at this location
                for drawer in drawer_names:
                    try:
                        print(f"[Exploration] Attempting to push {drawer} at {loc}")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        if done:
                            print(f"[Exploration] Successfully pushed {drawer} at {loc}")
                        else:
                            print(f"[Exploration] Could not push {drawer} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")

                # Try to place each object in each drawer at this location
                for obj in object_names:
                    for drawer in drawer_names:
                        try:
                            print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc}")
                            obs, reward, done = execute_place(env, task, obj, drawer, loc)
                            if done:
                                print(f"[Exploration] Successfully placed {obj} in {drawer} at {loc}")
                            else:
                                print(f"[Exploration] Could not place {obj} in {drawer} at {loc}")
                        except Exception as e:
                            print(f"[Exploration] execute_place failed for {obj} in {drawer} at {loc}: {e}")

                # Try to sweep each object at this location
                for obj in object_names:
                    try:
                        print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Successfully swept {obj} at {loc}")
                        else:
                            print(f"[Exploration] Could not sweep {obj} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

                # Try to use gripper (if applicable)
                try:
                    print(f"[Exploration] Attempting to use gripper at {loc}")
                    obs, reward, done = execute_gripper(env, task)
                    if done:
                        print(f"[Exploration] Successfully used gripper at {loc}")
                    else:
                        print(f"[Exploration] Could not use gripper at {loc}")
                except Exception as e:
                    print(f"[Exploration] execute_gripper failed at {loc}: {e}")

            except Exception as e:
                print(f"[Exploration] Error during exploration at {loc}: {e}")

        print("[Exploration] Exploration phase complete. Review logs to identify missing predicates or failed actions.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()