# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We need to explore the state of drawer1 to determine if it is closed.
        # Use only available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For this example, let's assume the following object names and locations:
        # - drawer1: the drawer to be explored
        # - handle1: the handle of drawer1
        # - robot starts at 'home' location
        # - drawer1 is at location 'drawer_area'
        # - handle1 is at location 'drawer_area'
        # These names should match those in your environment and object_positions.

        # You may need to adjust these names based on your actual environment.
        robot_location = 'home'
        drawer_location = 'drawer_area'
        drawer_name = 'drawer1'
        handle_name = 'handle1'

        # Step 1: Move robot to the drawer area if not already there
        try:
            if positions.get('robot') != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 2: Try to push the drawer (attempt to close it) to check if it is already closed
        # If the drawer is already closed, this action may fail or have no effect.
        try:
            print(f"[Exploration] Attempting to push (close) {drawer_name}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            print(f"[Exploration] execute_push called on {drawer_name}")
        except Exception as e:
            print(f"[Exploration] Error during execute_push: {e}")

        # Step 3: Try to pull (open) the drawer to check if it is locked or closed
        # First, pick the handle if needed
        try:
            print(f"[Exploration] Attempting to pick handle {handle_name}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
            print(f"[Exploration] execute_pick called on {handle_name}")
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        try:
            print(f"[Exploration] Attempting to pull (open) {drawer_name} using {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            print(f"[Exploration] execute_pull called on {drawer_name} with {handle_name}")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")

        # At this point, the robot has attempted to close and open the drawer.
        # Based on the feedback and the result of these actions, the code can infer the state of the drawer (closed/open/locked).
        # The missing predicate (drawer-closed drawer1) can be confirmed by the success/failure of execute_push and execute_pull.

        # === Main Task Plan (Oracle Plan Execution) ===
        # After exploration, proceed with the main plan using only available skills.
        # Example: pick an object from the floor and place it in the drawer.

        # For demonstration, let's assume there is an object 'obj1' on the floor at 'drawer_area'
        obj_name = 'obj1'
        try:
            print(f"[Task] Attempting to pick {obj_name} from floor at {drawer_location}")
            obs, reward, done = execute_pick(env, task, obj_name, drawer_location)
            print(f"[Task] execute_pick called on {obj_name}")
        except Exception as e:
            print(f"[Task] Error during execute_pick: {e}")

        # Place the object in the drawer (assume drawer is now open)
        try:
            print(f"[Task] Attempting to place {obj_name} into {drawer_name} at {drawer_location}")
            obs, reward, done = execute_place(env, task, obj_name, drawer_name, drawer_location)
            print(f"[Task] execute_place called for {obj_name} into {drawer_name}")
        except Exception as e:
            print(f"[Task] Error during execute_place: {e}")

        # Optionally, close the drawer after placing the object
        try:
            print(f"[Task] Attempting to push (close) {drawer_name}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            print(f"[Task] execute_push called on {drawer_name}")
        except Exception as e:
            print(f"[Task] Error during execute_push: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
