# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover missing predicates.
        # For this, we attempt to execute all available skills in a safe, exploratory manner.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to execute each skill with plausible arguments to see if any precondition fails
        # and to help identify missing predicates.
        # This is a generic exploration loop; in a real scenario, you would use more specific logic.

        print("[Exploration] Starting exploration to identify missing predicates...")

        # Example: Try to pick up each object on the floor at its location
        for obj_name, obj_info in positions.items():
            try:
                # Try execute_pick if the object is on the floor
                if obj_info.get('on_floor', False):
                    print(f"[Exploration] Trying execute_pick on {obj_name} at {obj_info['location']}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name,
                        obj_info['location']
                    )
                    if done:
                        print(f"[Exploration] Successfully picked {obj_name}.")
                        break
            except Exception as e:
                print(f"[Exploration] execute_pick failed for {obj_name}: {e}")

        # Try to open each drawer if possible
        for drawer_name, drawer_info in positions.items():
            try:
                if drawer_info.get('type', '') == 'drawer' and drawer_info.get('closed', False):
                    handle_name = drawer_info.get('handle')
                    if handle_name:
                        print(f"[Exploration] Trying execute_pull on {drawer_name} using handle {handle_name} at {drawer_info['location']}")
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            drawer_name,
                            handle_name,
                            drawer_info['location']
                        )
                        if done:
                            print(f"[Exploration] Successfully pulled {drawer_name}.")
                            break
            except Exception as e:
                print(f"[Exploration] execute_pull failed for {drawer_name}: {e}")

        # Try to move the robot between locations
        locations = set()
        for obj in positions.values():
            loc = obj.get('location')
            if loc:
                locations.add(loc)
        locations = list(locations)
        if len(locations) > 1:
            try:
                print(f"[Exploration] Trying execute_go from {locations[0]} to {locations[1]}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    locations[0],
                    locations[1]
                )
                if done:
                    print(f"[Exploration] Successfully moved from {locations[0]} to {locations[1]}.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # Try to place an object in a drawer if possible
        for obj_name, obj_info in positions.items():
            if obj_info.get('holding', False):
                for drawer_name, drawer_info in positions.items():
                    if drawer_info.get('type', '') == 'drawer' and drawer_info.get('open', False):
                        try:
                            print(f"[Exploration] Trying execute_place for {obj_name} into {drawer_name} at {drawer_info['location']}")
                            obs, reward, done = execute_place(
                                env,
                                task,
                                obj_name,
                                drawer_name,
                                drawer_info['location']
                            )
                            if done:
                                print(f"[Exploration] Successfully placed {obj_name} in {drawer_name}.")
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_place failed for {obj_name} in {drawer_name}: {e}")

        # Try other available skills for completeness
        try:
            print("[Exploration] Trying execute_push on all drawers.")
            for drawer_name, drawer_info in positions.items():
                if drawer_info.get('type', '') == 'drawer' and drawer_info.get('open', False):
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name,
                        drawer_info['location']
                    )
                    if done:
                        print(f"[Exploration] Successfully pushed {drawer_name}.")
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")

        try:
            print("[Exploration] Trying execute_sweep on all objects on the floor.")
            for obj_name, obj_info in positions.items():
                if obj_info.get('on_floor', False):
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        obj_name,
                        obj_info['location']
                    )
                    if done:
                        print(f"[Exploration] Successfully swept {obj_name}.")
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        try:
            print("[Exploration] Trying execute_gripper.")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Successfully executed gripper action.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for failed preconditions or missing predicates.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # At this point, after exploration, you would normally execute the oracle plan.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
