# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Find the missing predicate (robot-at drawer-area) ---
        # The feedback indicates that (robot-at drawer-area) is missing or not being set.
        # We will use the available skills to move the robot to 'drawer-area' and check the state.

        # Try to find the location key for 'drawer-area'
        drawer_area_key = None
        for key in positions.keys():
            if 'drawer-area' in key or 'drawer' in key:
                drawer_area_key = key
                break

        if drawer_area_key is None:
            print("[Exploration] Could not find 'drawer-area' in object positions. Available keys:", list(positions.keys()))
            # Fallback: try a default name
            drawer_area_key = 'drawer-area'

        # Try to find the robot's current location
        robot_location = None
        for key in positions.keys():
            if 'robot' in key and 'area' in key:
                robot_location = key
                break
        if robot_location is None:
            # Fallback: try a default name
            robot_location = 'robot-home'

        # Print current positions for debugging
        print("[Exploration] Object positions:", positions)
        print("[Exploration] Robot location key:", robot_location)
        print("[Exploration] Drawer area key:", drawer_area_key)

        # --- Use execute_go to move robot to drawer-area ---
        # The skill 'execute_go' expects (from, to) location names
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {drawer_area_key} using execute_go.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_area_key,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {drawer_area_key}.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")

        # --- Check if the predicate (robot-at drawer-area) is now true ---
        # This would typically be checked via the environment's state or observation
        # For demonstration, print the observation or state
        try:
            current_obs = task.get_observation()
            print("[Exploration] Current observation after move:", current_obs)
        except Exception as e:
            print(f"[Exploration] Exception getting observation: {e}")

        # --- Exploration: Try other skills if needed to further probe the state ---
        # For example, try execute_gripper or execute_sweep to see if they affect predicates
        try:
            print("[Exploration] Executing execute_gripper to probe robot-free/hand state.")
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=50,
                timeout=5.0
            )
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        # --- End of Exploration Phase ---

        print("[Exploration] Exploration phase complete. If (robot-at drawer-area) is still missing, check domain/observation definitions.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
