# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names based on the example problem ===
        # These should match the names in your PDDL problem and environment
        dice_names = ['dice1', 'dice2']
        handle_names = ['handle1', 'handle2', 'handle3']
        drawer_names = ['drawer1', 'drawer2', 'drawer3']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # Map handles to drawers
        handle_to_drawer = {
            'handle1': 'drawer1',
            'handle2': 'drawer2',
            'handle3': 'drawer3'
        }

        # Initial robot location
        robot_location = 'ready-pose'

        # === Exploration Phase: Identify Missing Predicate (lock-known) ===
        # The feedback and exploration domain suggest we need to check the lock state of drawers.
        # We'll attempt to pull each handle to determine if the drawer is locked or unlocked.
        # This is a safe exploration step before attempting to open drawers.

        print("[Exploration] Checking lock state of drawers via handles...")
        for handle in handle_names:
            drawer = handle_to_drawer[handle]
            handle_pos = positions.get(handle, None)
            if handle_pos is None:
                print(f"[Warning] Handle {handle} position not found, skipping.")
                continue
            # Move to handle location if not already there
            if robot_location != handle_pos['location']:
                try:
                    obs, reward, done = execute_go(env, task, robot_location, handle_pos['location'])
                    robot_location = handle_pos['location']
                except Exception as e:
                    print(f"[Error] Failed to move to {handle}: {e}")
                    continue
            # Pick the handle to hold it
            try:
                obs, reward, done = execute_pick(env, task, handle, robot_location)
            except Exception as e:
                print(f"[Error] Failed to pick handle {handle}: {e}")
                continue
            # Try to pull the handle to check lock state (exploration)
            try:
                obs, reward, done = execute_pull(env, task, drawer, handle, robot_location)
                print(f"[Exploration] Pulled {handle} for {drawer}: lock-known predicate updated.")
            except Exception as e:
                print(f"[Exploration] Could not pull {handle} for {drawer} (may be locked): {e}")
            # Place handle back on floor (if needed)
            try:
                obs, reward, done = execute_place(env, task, handle, drawer, robot_location)
            except Exception as e:
                print(f"[Warning] Could not place handle {handle} after exploration: {e}")

        # === Main Task Logic: Put dice1 and dice2 into drawer1 ===
        # According to the initial state, drawer1 is open and empty, so we can place objects directly.

        # Move to ready-pose if not already there
        if robot_location != 'ready-pose':
            try:
                obs, reward, done = execute_go(env, task, robot_location, 'ready-pose')
                robot_location = 'ready-pose'
            except Exception as e:
                print(f"[Error] Failed to move to ready-pose: {e}")

        # For each dice, pick from floor and place into drawer1
        for dice in dice_names:
            dice_pos = positions.get(dice, None)
            if dice_pos is None:
                print(f"[Warning] Dice {dice} position not found, skipping.")
                continue
            # Move to dice location if not already there
            if robot_location != dice_pos['location']:
                try:
                    obs, reward, done = execute_go(env, task, robot_location, dice_pos['location'])
                    robot_location = dice_pos['location']
                except Exception as e:
                    print(f"[Error] Failed to move to {dice}: {e}")
                    continue
            # Pick the dice
            try:
                obs, reward, done = execute_pick(env, task, dice, robot_location)
            except Exception as e:
                print(f"[Error] Failed to pick {dice}: {e}")
                continue
            # Move to drawer1 location (assume drawer1 is at 'drawer-area')
            drawer1_pos = positions.get('drawer1', {'location': 'drawer-area'})
            if robot_location != drawer1_pos['location']:
                try:
                    obs, reward, done = execute_go(env, task, robot_location, drawer1_pos['location'])
                    robot_location = drawer1_pos['location']
                except Exception as e:
                    print(f"[Error] Failed to move to drawer1: {e}")
                    continue
            # Place the dice into drawer1
            try:
                obs, reward, done = execute_place(env, task, dice, 'drawer1', robot_location)
                print(f"[Task] Placed {dice} into drawer1.")
            except Exception as e:
                print(f"[Error] Failed to place {dice} into drawer1: {e}")
                continue

        # After placing both dice, close drawer1 for safety
        try:
            obs, reward, done = execute_push(env, task, 'drawer1', robot_location)
            print("[Task] Closed drawer1 after placing dice.")
        except Exception as e:
            print(f"[Warning] Could not close drawer1: {e}")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
