# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The predicate (robot-at drawer-area) is missing or not being set as expected.
        # We need to explore the environment to determine if the robot can be at 'drawer-area'
        # and which skill/action is responsible for setting this predicate.

        # We will use the available skills to move the robot to 'drawer-area' and check the result.

        # Step 1: Determine current robot location and the 'drawer-area' location
        # For this example, we assume 'drawer-area' is a key in positions
        # and that the robot's current location is also available.

        # Try to get the robot's current location and the drawer-area position
        try:
            robot_location = None
            drawer_area_location = None

            # Find robot's current location
            for key in positions:
                if 'robot' in key and 'location' in key:
                    robot_location = positions[key]
                    break
            # Fallback: try a default key
            if robot_location is None and 'robot_location' in positions:
                robot_location = positions['robot_location']

            # Find drawer-area location
            if 'drawer-area' in positions:
                drawer_area_location = positions['drawer-area']
            elif 'drawer_area' in positions:
                drawer_area_location = positions['drawer_area']
            else:
                # Try to find a key that matches
                for key in positions:
                    if 'drawer' in key and 'area' in key:
                        drawer_area_location = positions[key]
                        break

            if robot_location is None or drawer_area_location is None:
                print("[Exploration] Could not determine robot or drawer-area location from positions.")
                print("[Exploration] positions keys:", list(positions.keys()))
                return

            print(f"[Exploration] Robot location: {robot_location}")
            print(f"[Exploration] Drawer-area location: {drawer_area_location}")

            # Step 2: Use the 'execute_go' skill to move the robot to the drawer-area
            # The skill signature is assumed to be: execute_go(env, task, from_location, to_location, ...)
            print("[Exploration] Attempting to move robot to drawer-area using execute_go...")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=drawer_area_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] execute_go completed.")
            except Exception as e:
                print("[Exploration] Error during execute_go:", e)
                return

            # Step 3: After moving, check if the predicate (robot-at drawer-area) is now true
            # This may require checking the environment state or observation
            # For demonstration, we print the observation and look for the predicate
            print("[Exploration] Checking if (robot-at drawer-area) is now true in the observation/state...")
            try:
                current_obs = task.get_observation()
                # Print or log the observation for debugging
                print("[Exploration] Current observation after move:", current_obs)
                # If the observation is a dict or has a 'predicates' field, check for the predicate
                if isinstance(current_obs, dict):
                    predicates = current_obs.get('predicates', [])
                    if any('robot-at' in p and 'drawer-area' in p for p in predicates):
                        print("[Exploration] Predicate (robot-at drawer-area) is now true.")
                    else:
                        print("[Exploration] Predicate (robot-at drawer-area) is still missing.")
                else:
                    print("[Exploration] Unable to parse observation for predicates.")
            except Exception as e:
                print("[Exploration] Error while checking observation:", e)

        except Exception as e:
            print("[Exploration] Error during exploration phase:", e)

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan after exploration as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()