# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('location')]
        location_names = [name for name in positions if name.startswith('location')]

        # For demonstration, we will try to interact with all objects and drawers using available skills.
        # We will attempt to pick, place, pull, push, sweep, rotate, go, and gripper actions as appropriate.

        # 1. Try to pick up all objects on the floor at their locations
        for obj in object_names:
            try:
                obj_info = positions[obj]
                if 'location' in obj_info:
                    loc = obj_info['location']
                else:
                    # Fallback: try to find a location key
                    loc = None
                    for key in obj_info:
                        if 'location' in key:
                            loc = obj_info[key]
                            break
                if loc is not None:
                    print(f"[Exploration] Attempting to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {loc}")
                        break
            except Exception as e:
                print(f"[Exploration] Could not pick {obj}: {e}")

        # 2. Try to open all drawers by pulling their handles (if any)
        for obj in object_names:
            try:
                obj_info = positions[obj]
                if 'drawer' in obj_info:
                    drawer = obj_info['drawer']
                    handle = obj  # Assume the object is the handle
                    if 'location' in obj_info:
                        loc = obj_info['location']
                    else:
                        loc = None
                        for key in obj_info:
                            if 'location' in key:
                                loc = obj_info[key]
                                break
                    if loc is not None:
                        print(f"[Exploration] Attempting to pull handle {handle} of drawer {drawer} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print(f"[Exploration] Successfully pulled handle {handle} of drawer {drawer} at {loc}")
                            break
            except Exception as e:
                print(f"[Exploration] Could not pull handle {obj}: {e}")

        # 3. Try to push (close) all drawers at their locations
        for obj in object_names:
            try:
                obj_info = positions[obj]
                if 'drawer' in obj_info:
                    drawer = obj_info['drawer']
                    if 'location' in obj_info:
                        loc = obj_info['location']
                    else:
                        loc = None
                        for key in obj_info:
                            if 'location' in key:
                                loc = obj_info[key]
                                break
                    if loc is not None:
                        print(f"[Exploration] Attempting to push (close) drawer {drawer} at {loc}")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        if done:
                            print(f"[Exploration] Successfully pushed (closed) drawer {drawer} at {loc}")
                            break
            except Exception as e:
                print(f"[Exploration] Could not push (close) drawer {obj}: {e}")

        # 4. Try to place any held object into any open drawer at the same location
        for obj in object_names:
            try:
                obj_info = positions[obj]
                if 'drawer' in obj_info and 'location' in obj_info:
                    drawer = obj_info['drawer']
                    loc = obj_info['location']
                    print(f"[Exploration] Attempting to place {obj} into drawer {drawer} at {loc}")
                    obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into drawer {drawer} at {loc}")
                        break
            except Exception as e:
                print(f"[Exploration] Could not place {obj} into drawer: {e}")

        # 5. Try to sweep all objects at their locations
        for obj in object_names:
            try:
                obj_info = positions[obj]
                if 'location' in obj_info:
                    loc = obj_info['location']
                    print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully swept {obj} at {loc}")
                        break
            except Exception as e:
                print(f"[Exploration] Could not sweep {obj}: {e}")

        # 6. Try to move (go) between all pairs of locations
        for i in range(len(location_names)):
            for j in range(len(location_names)):
                if i != j:
                    from_loc = location_names[i]
                    to_loc = location_names[j]
                    try:
                        print(f"[Exploration] Attempting to go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                            break
                    except Exception as e:
                        print(f"[Exploration] Could not go from {from_loc} to {to_loc}: {e}")

        # 7. Try to use the gripper (if available)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Successfully used gripper")
        except Exception as e:
            print(f"[Exploration] Could not use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()