# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (robot-at floor) is missing or problematic.
        # We'll attempt to use available skills to explore and identify the missing predicate.

        # For demonstration, let's assume the robot starts at some location and we want to check if it can move to 'floor'
        # and if the predicate (robot-at floor) is recognized by the environment.

        # We'll use the available skill 'execute_go' to try moving to 'floor' if possible.
        # If an exception or failure occurs, we infer that the predicate is missing or not handled.

        # Get all locations from positions (assuming keys like 'floor', 'drawer', etc.)
        location_names = [name for name in positions.keys() if 'floor' in name or 'location' in name or 'room' in name or 'drawer' in name]
        # Fallback: just try 'floor' if present
        if 'floor' in positions:
            floor_location = 'floor'
        elif len(location_names) > 0:
            floor_location = location_names[0]
        else:
            floor_location = None

        # Try to get the robot's current location
        robot_location = None
        for name in positions:
            if 'robot' in name:
                robot_location = positions[name]
                break

        # If not found, just pick a default
        if robot_location is None and len(location_names) > 0:
            robot_location = positions[location_names[0]]

        # Try to execute 'execute_go' to 'floor' and catch any errors
        exploration_success = False
        if floor_location is not None and robot_location is not None:
            try:
                print(f"[Exploration] Attempting to move robot to '{floor_location}' using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=floor_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to '{floor_location}'. Checking for predicate (robot-at {floor_location})...")
                # Here, you would check the environment state for the predicate, but since we can't access PDDL state directly,
                # we assume success if no exception is raised.
                exploration_success = True
            except Exception as e:
                print(f"[Exploration] Failed to move robot to '{floor_location}': {e}")
                exploration_success = False
        else:
            print("[Exploration] Could not determine robot or floor location for exploration.")

        if not exploration_success:
            print("[Exploration] The predicate (robot-at floor) may be missing or not handled in the environment.")
        else:
            print("[Exploration] The predicate (robot-at floor) appears to be present and handled.")

        # === Main Task Plan ===
        # Here you would continue with the oracle plan, using only the available skills.
        # For demonstration, let's try a sequence of available skills if possible.

        # Example: Try to pick up an object on the floor if present
        object_on_floor = None
        for obj_name in positions:
            if 'object' in obj_name or 'ball' in obj_name:
                object_on_floor = obj_name
                break

        if object_on_floor and floor_location:
            try:
                print(f"[Task] Attempting to pick up '{object_on_floor}' from '{floor_location}' using execute_pick.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=object_on_floor,
                    location=floor_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Task] Picked up '{object_on_floor}'.")
            except Exception as e:
                print(f"[Task] Failed to pick up '{object_on_floor}': {e}")

        # You can continue with other skills as needed, e.g., execute_place, execute_push, etc.
        # For each, use try/except to handle errors gracefully.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
