# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (robot-at drawer-area)
        # The goal is to explore and determine if the predicate (robot-at drawer-area) is missing or not recognized.
        # We will attempt to move the robot to 'drawer-area' using available skills and observe the result.

        # Try to find a location named 'drawer-area' in the positions dictionary
        drawer_area_key = None
        for key in positions:
            if 'drawer' in key and 'area' in key:
                drawer_area_key = key
                break
        if drawer_area_key is None:
            # Fallback: try common variants
            for key in positions:
                if 'drawer' in key:
                    drawer_area_key = key
                    break

        if drawer_area_key is None:
            print("[Exploration] Could not find 'drawer-area' in object positions. Exploration aborted.")
            return

        drawer_area_pos = positions[drawer_area_key]
        print(f"[Exploration] Identified drawer area as '{drawer_area_key}' at position {drawer_area_pos}")

        # Find current robot location
        robot_location_key = None
        for key in positions:
            if 'robot' in key or 'base' in key:
                robot_location_key = key
                break
        if robot_location_key is None:
            # Fallback: try to infer from observation or use a default
            robot_location_key = 'robot_base'
        robot_location_pos = positions.get(robot_location_key, None)

        # Try to move the robot to the drawer area using the available skill
        # According to available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # The move action is 'execute_go'
        # We need to provide 'from' and 'to' locations (names, not positions)

        # For demonstration, let's assume location names are keys in positions
        from_location = robot_location_key
        to_location = drawer_area_key

        print(f"[Exploration] Attempting to move robot from '{from_location}' to '{to_location}' using execute_go.")

        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location,
                to_location
            )
            print(f"[Exploration] execute_go returned: done={done}, reward={reward}")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")
            print("[Exploration] This may indicate a missing predicate or misconfigured domain (e.g., (robot-at drawer-area) missing).")
            return

        # After moving, check if the robot is at the drawer area
        # This would typically be checked via the environment's state or observation
        # For demonstration, print the observation or check for the predicate
        try:
            current_obs = task.get_observation()
            print(f"[Exploration] Current observation after move: {current_obs}")
        except Exception as e:
            print(f"[Exploration] Exception when getting observation: {e}")

        print("[Exploration] If the robot failed to move or the predicate (robot-at drawer-area) is not updated, the predicate may be missing in the domain/problem definition.")

        # --- End of Exploration Phase ---

        # TODO: Continue with the main oracle plan after exploration, using only predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()