# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The missing predicate is likely the robot's location at the light switch.
        # We need to explore the environment to discover the location of the light switch.

        # Try to move the robot to all known locations and check if any correspond to the light switch.
        # Use only available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assume positions dict contains all known locations, including possible 'light_switch'
        location_names = [name for name in positions.keys() if 'location' in name or 'switch' in name or 'drawer' in name or 'room' in name]
        robot_current_location = None

        # Try to determine the robot's current location
        for name in positions.keys():
            if 'robot' in name:
                robot_current_location = positions[name]
                break

        # If not found, try to infer from observation or set a default
        if robot_current_location is None:
            # Fallback: pick the first location as starting point
            if len(location_names) > 0:
                robot_current_location = positions[location_names[0]]
                robot_location_name = location_names[0]
            else:
                print("[Exploration] No known locations found in positions.")
                robot_location_name = None
        else:
            robot_location_name = None
            for loc in location_names:
                if positions[loc] == robot_current_location:
                    robot_location_name = loc
                    break

        # If we don't know the robot's location name, just pick the first
        if robot_location_name is None and len(location_names) > 0:
            robot_location_name = location_names[0]

        # Now, try to move to each location using execute_go and see if any is the light switch
        found_light_switch = False
        for loc in location_names:
            if loc == robot_location_name:
                continue  # Already at this location
            try:
                print(f"[Exploration] Attempting to move robot from {robot_location_name} to {loc} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location_name,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                # After moving, check if this location is the light switch
                # This can be done by checking the feedback or by querying the environment state
                # For this exploration, we assume that if the move succeeds, we can check if the location is 'light_switch'
                if 'light_switch' in loc or 'switch' in loc:
                    print(f"[Exploration] Robot is now at the light switch location: {loc}")
                    found_light_switch = True
                    # Optionally, perform the switch action if needed
                    try:
                        print("[Exploration] Attempting to execute_push_switch at light switch.")
                        obs, reward, done = execute_push_switch(
                            env,
                            task,
                            switch_name=loc,
                            location=loc,
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                        print("[Exploration] Successfully executed push switch.")
                    except Exception as e:
                        print(f"[Exploration] Failed to execute push switch at {loc}: {e}")
                    break
                else:
                    # Not the light switch, continue exploring
                    robot_location_name = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

        if not found_light_switch:
            print("[Exploration] Could not find the light switch location during exploration.")

        # === End of Exploration Phase ===

        # At this point, the missing predicate (robot-at light_switch) should be discovered or confirmed.
        # Further task logic (oracle plan) would proceed here, using only the available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
