# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The robot must discover the location of the light switch and the predicate that describes it.
        # We will use available skills to explore the environment and check for the presence of the light switch.

        # Try to find the light switch location among known positions
        light_switch_pos = None
        for obj_name, pos in positions.items():
            if 'light_switch' in obj_name or 'switch' in obj_name or 'light' in obj_name:
                light_switch_pos = pos
                light_switch_name = obj_name
                break

        if light_switch_pos is None:
            print("[Exploration] Light switch not found in object positions. Attempting to explore all locations.")
            # Fallback: Try all known locations to find the light switch
            for obj_name, pos in positions.items():
                try:
                    # Try to use execute_go to move to the location and see if it is a switch
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=None,  # Let the skill handle current location
                        to_location=pos,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    # After moving, try to execute_push_switch at this location
                    obs, reward, done = execute_push_switch(
                        env,
                        task,
                        switch_name=obj_name,
                        location=pos,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] Successfully pushed switch at {obj_name}.")
                    light_switch_pos = pos
                    light_switch_name = obj_name
                    break
                except Exception as e:
                    # Not a switch or failed to push, continue exploring
                    continue

        if light_switch_pos is None:
            print("[Exploration] Could not find or push any light switch. Exploration failed.")
            return

        print(f"[Exploration] Light switch found: {light_switch_name} at {light_switch_pos}")

        # === Predicate Discovery ===
        # The missing predicate is likely (robot-at light_switch)
        # We can confirm by checking if the robot is at the light switch location after moving
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=None,
                to_location=light_switch_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to light switch location: {light_switch_pos}")
            # Now, try to push the switch to turn on the light
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name=light_switch_name,
                location=light_switch_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Successfully executed push switch. Room should now be bright.")
        except Exception as e:
            print(f"[Exploration] Failed to move to or push the light switch: {e}")
            return

        # === End of Exploration Phase ===
        # At this point, the missing predicate (robot-at light_switch) is discovered and the room is bright.
        # The rest of the oracle plan can proceed as normal using the available skills.

        # === Example: Continue with Oracle Plan (if provided) ===
        # (This is a placeholder for further plan steps, e.g., pick, place, etc.)
        # For demonstration, we print the current state.
        print("[Task] Exploration complete. Ready to execute the main oracle plan.")

        # Example: If the next step is to pick an object after the room is bright
        # You would use:
        # obs, reward, done = execute_pick(env, task, object_name, location, ...)

        # TODO: Insert further oracle plan steps here as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()