# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The goal is to identify which predicate is missing by attempting available skills and observing feedback.
        # We will attempt to execute each available skill in a safe, exploratory manner.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt each skill with plausible arguments.
        # In a real scenario, you would use the actual objects, drawers, locations, etc.
        # Here, we use the positions dictionary to get object names and locations.

        # Extract object, drawer, and location names from positions (if available)
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'place' in name]

        # Fallbacks if names are not found
        if not object_names:
            object_names = list(positions.keys())
        if not location_names:
            location_names = list(positions.keys())

        # Use the first available object, drawer, and location for exploration
        obj = object_names[0] if object_names else None
        drawer = drawer_names[0] if drawer_names else None
        loc = location_names[0] if location_names else None

        # Try each skill and catch exceptions to infer missing predicates
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # execute_pick(env, obj, loc)
                    if obj and loc:
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"[Exploration] execute_pick({obj}, {loc}) succeeded.")
                elif skill == 'execute_place':
                    # execute_place(env, obj, drawer, loc)
                    if obj and drawer and loc:
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] execute_place({obj}, {drawer}, {loc}) succeeded.")
                elif skill == 'execute_push':
                    # execute_push(env, drawer, loc)
                    if drawer and loc:
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"[Exploration] execute_push({drawer}, {loc}) succeeded.")
                elif skill == 'execute_pull':
                    # execute_pull(env, drawer, obj, loc)
                    if drawer and obj and loc:
                        obs, reward, done = execute_pull(env, task, drawer, obj, loc)
                        print(f"[Exploration] execute_pull({drawer}, {obj}, {loc}) succeeded.")
                elif skill == 'execute_sweep':
                    # execute_sweep(env, obj, loc)
                    if obj and loc:
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"[Exploration] execute_sweep({obj}, {loc}) succeeded.")
                elif skill == 'execute_rotate':
                    # execute_rotate(env, obj, loc)
                    if obj and loc:
                        obs, reward, done = execute_rotate(env, task, obj, loc)
                        print(f"[Exploration] execute_rotate({obj}, {loc}) succeeded.")
                elif skill == 'execute_go':
                    # execute_go(env, from_loc, to_loc)
                    if len(location_names) >= 2:
                        from_loc = location_names[0]
                        to_loc = location_names[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"[Exploration] execute_go({from_loc}, {to_loc}) succeeded.")
                elif skill == 'execute_gripper':
                    # execute_gripper(env)
                    obs, reward, done = execute_gripper(env, task)
                    print(f"[Exploration] execute_gripper() succeeded.")
            except Exception as e:
                print(f"[Exploration] {skill} failed with error: {e}")

        print("[Exploration] Exploration phase complete. Check above for failed skills and possible missing predicates.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this task, the focus is on exploration to find missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()