# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (room-bright) is missing or not satisfied.
        # According to the domain, most actions require (room-bright) as a precondition.
        # The only action that can change (room-dark) to (room-bright) is execute_push_switch.
        # Therefore, we must ensure the room is bright before any further actions.

        # Try to execute execute_push_switch if the room is dark.
        # We assume the switch and its location are known from the environment.
        # We'll attempt to find a switch and its location from the positions dictionary.

        switch_name = None
        switch_location = None

        # Try to find a switch in the positions dictionary
        for obj_name, obj_info in positions.items():
            if 'switch' in obj_name.lower():
                switch_name = obj_name
                # Try to get location info
                if isinstance(obj_info, dict) and 'location' in obj_info:
                    switch_location = obj_info['location']
                else:
                    # If only position is available, use it as location
                    switch_location = obj_info
                break

        # If no switch found, try a default name
        if switch_name is None:
            switch_name = 'switch'
            switch_location = None  # Will try to use a default location if needed

        # If location is not found, try to find a plausible location
        if switch_location is None:
            # Try to find a location key in positions
            for obj_name, obj_info in positions.items():
                if 'location' in obj_name.lower():
                    switch_location = obj_name
                    break
            # If still not found, use a default string
            if switch_location is None:
                switch_location = 'room_center'

        # Now, try to execute execute_push_switch to turn on the light
        try:
            print(f"[Exploration] Attempting to turn on the light using switch '{switch_name}' at location '{switch_location}'")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name,
                switch_location
            )
            print("[Exploration] Light switch action executed. Room should now be bright.")
        except Exception as e:
            print(f"[Exploration] Failed to execute execute_push_switch: {e}")

        # === Continue with Task Plan (Example) ===
        # At this point, the room should be bright and further actions can be performed.
        # You can now proceed with the oracle plan, e.g., pick, place, move, etc.
        # For demonstration, we show how to pick up an object if available.

        # Try to find an object on the floor to pick up
        object_to_pick = None
        object_location = None
        for obj_name, obj_info in positions.items():
            if 'object' in obj_name.lower() or 'ball' in obj_name.lower() or 'cube' in obj_name.lower():
                object_to_pick = obj_name
                if isinstance(obj_info, dict) and 'location' in obj_info:
                    object_location = obj_info['location']
                else:
                    object_location = obj_info
                break

        if object_to_pick is not None and object_location is not None:
            try:
                print(f"[Task] Attempting to pick up '{object_to_pick}' at location '{object_location}'")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_to_pick,
                    object_location
                )
                print(f"[Task] Picked up '{object_to_pick}'.")
            except Exception as e:
                print(f"[Task] Failed to execute execute_pick: {e}")
        else:
            print("[Task] No suitable object found to pick up.")

        # Additional plan steps (place, move, etc.) would go here, using only predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
