# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The robot needs to know if it can reach the light switch location.
        # We'll attempt to move the robot to the light_switch location using available skills.
        # If the action fails, we infer the missing predicate.

        # Assume 'light_switch' is a location in the environment.
        # Try to find the robot's current location and the light_switch location.
        robot_location = None
        light_switch_location = None

        # Try to find the robot's current location and the light_switch location from positions
        for name, pos in positions.items():
            if 'robot' in name or 'robby' in name:
                robot_location = name
            if 'light_switch' in name or 'switch' in name:
                light_switch_location = name

        # If not found, use default names
        if robot_location is None:
            robot_location = 'robot_base'  # fallback, may need adjustment
        if light_switch_location is None:
            light_switch_location = 'light_switch'

        print(f"[Exploration] Robot location: {robot_location}")
        print(f"[Exploration] Light switch location: {light_switch_location}")

        # Try to move to the light switch using execute_go
        # execute_go(?from, ?to)
        exploration_success = False
        try:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {light_switch_location} using execute_go.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] execute_go succeeded.")
            exploration_success = True
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")
            exploration_success = False

        # If exploration failed, the missing predicate is likely (robot-at light_switch)
        if not exploration_success:
            print("[Exploration] Missing predicate detected: (robot-at light_switch)")
            # Optionally, log or handle this information for further planning

        # === Main Task Plan Execution ===
        # Example: Turn on the light if the room is dark and robot is at the light switch
        # Use execute_push_switch(?s, ?p)
        try:
            print("[Task] Attempting to turn on the light using execute_push_switch.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch=light_switch_location,
                location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Task] Light switch toggled (room should now be bright).")
        except Exception as e:
            print(f"[Task] Failed to execute_push_switch: {e}")

        # Continue with further plan steps as needed, using only available skills
        # For example, pick up an object if required:
        # try:
        #     obs, reward, done = execute_pick(env, task, object_name, location, ...)
        # except Exception as e:
        #     print(f"Failed to pick object: {e}")

        # TODO: Add further plan steps as required by the oracle plan and task

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()