# run_skeleton_task.py (Completed for "put two dice in the open drawer" with exploration for missing predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'dice1': (x1, y1, z1), 'dice2': (x2, y2, z2), 'drawer1': (xd, yd, zd), ...}

        # === Object/Location Names (assumed from context) ===
        # You may need to adjust these names to match your environment
        dice_names = ['dice1', 'dice2']
        drawer_name = 'drawer1'
        light_switch_name = 'light_switch'
        robot_location = None
        drawer_location = None
        light_switch_location = None

        # Try to infer locations from positions or from the environment
        try:
            # If positions dict includes location mapping, use it
            if 'drawer1' in positions:
                drawer_location = positions['drawer1']
            if 'light_switch' in positions:
                light_switch_location = positions['light_switch']
            # For robot, you may need to query the environment or use a default
            if 'robot' in positions:
                robot_location = positions['robot']
        except Exception as e:
            print("[Warning] Could not extract all positions:", e)

        # === Exploration Phase: Check for missing predicates (e.g., room-bright) ===
        # The robot cannot pick/place in the dark. If the room is dark, turn on the light.
        # We'll try to execute_pick and if it fails, we assume the room is dark and turn on the light.

        # Helper: get current robot location (if available)
        def get_robot_location():
            # Try to get from positions, else fallback
            if 'robot' in positions:
                return positions['robot']
            # Fallback: ask the environment (if possible)
            try:
                return task.get_robot_location()
            except Exception:
                return None

        # Helper: get location of an object
        def get_object_location(obj_name):
            if obj_name in positions:
                return positions[obj_name]
            return None

        # Helper: get location name for a given object (if available)
        def get_location_name_for(obj_name):
            # If positions dict has mapping from object to location name
            if obj_name+'_location' in positions:
                return positions[obj_name+'_location']
            # Fallback: just use the object name
            return obj_name

        # === Step 1: Ensure the room is bright (if not, turn on the light) ===
        room_is_bright = False
        try:
            # Try a dry-run pick to see if room is bright
            obs, reward, done = execute_pick(env, task, dice_names[0], get_location_name_for(dice_names[0]))
            if done or reward > 0:
                room_is_bright = True
        except Exception as e:
            # If pick fails due to precondition, assume room is dark
            print("[Exploration] Pick failed, likely due to room being dark. Turning on the light.")
            # Move to light switch if not already there
            try:
                # If robot is not at light switch, move there
                robot_loc = get_robot_location()
                if robot_loc is not None and light_switch_location is not None and robot_loc != light_switch_location:
                    obs, reward, done = execute_go(env, task, robot_loc, light_switch_location)
                # Push the light switch
                obs, reward, done = execute_push_switch(env, task, light_switch_name, light_switch_location)
                room_is_bright = True
            except Exception as e2:
                print("[Error] Could not turn on the light:", e2)
                return  # Cannot proceed if light cannot be turned on

        # === Step 2: Move to dice and pick them up one by one ===
        for dice in dice_names:
            try:
                # Move to dice location if not already there
                dice_loc = get_location_name_for(dice)
                robot_loc = get_robot_location()
                if robot_loc != dice_loc:
                    obs, reward, done = execute_go(env, task, robot_loc, dice_loc)
                # Pick the dice
                obs, reward, done = execute_pick(env, task, dice, dice_loc)
                if done:
                    print(f"[Task] Picked up {dice}.")
            except Exception as e:
                print(f"[Error] Could not pick up {dice}:", e)
                continue

            # === Step 3: Move to drawer and place the dice ===
            try:
                # Move to drawer location if not already there
                robot_loc = get_robot_location()
                if robot_loc != drawer_location:
                    obs, reward, done = execute_go(env, task, robot_loc, drawer_location)
                # Place the dice in the drawer
                obs, reward, done = execute_place(env, task, dice, drawer_name, drawer_location)
                if done:
                    print(f"[Task] Placed {dice} in {drawer_name}.")
            except Exception as e:
                print(f"[Error] Could not place {dice} in {drawer_name}:", e)
                continue

        # === Task Complete ===
        print("[Task] All dice placed in the drawer (if possible).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()