# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The robot must discover the location of the light switch (or similar object).
        # We will attempt to move the robot to all known locations and check for the presence of the light switch.

        # Assume positions is a dict: {object_name: (x, y, z), ...}
        # Try to find the light switch location
        light_switch_location = None
        for obj_name in positions:
            if "light_switch" in obj_name or "switch" in obj_name:
                light_switch_location = obj_name
                break

        if light_switch_location is None:
            print("[Exploration] No light switch found in object positions. Exploration cannot proceed.")
            return

        # Now, try to move the robot to the light switch location using available skills
        # The available skill for moving is 'execute_go'
        # We need to know the robot's current location and the target location

        # Find robot's current location
        robot_location = None
        for obj_name in positions:
            if "robot" in obj_name or "base" in obj_name:
                robot_location = obj_name
                break

        # If robot location is not explicitly given, try to infer from observation
        if robot_location is None:
            # Fallback: try to get from observation or assume a default
            robot_location = "robot_base"

        # For the sake of this code, we assume the location names are keys in positions
        # and that the skill functions accept these names as parameters

        # Try to move to the light switch location
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} to check for missing predicate.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=light_switch_location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {light_switch_location}.")
        except Exception as e:
            print(f"[Exploration] Failed to move robot to {light_switch_location}: {e}")

        # After moving, try to interact with the switch if possible
        # Use 'execute_push_switch' if available
        try:
            print(f"[Exploration] Attempting to push the switch at {light_switch_location}.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name=light_switch_location,
                location=light_switch_location,
                approach_distance=0.10,
                max_steps=50,
                threshold=0.01,
                approach_axis='z',
                timeout=5.0
            )
            print(f"[Exploration] Switch at {light_switch_location} pushed.")
        except Exception as e:
            print(f"[Exploration] Failed to push switch at {light_switch_location}: {e}")

        # At this point, the robot should have discovered the missing predicate (robot-at light_switch)
        # or at least have interacted with the environment to reveal it.

        # === End of Exploration Phase ===

        # You can now proceed with the rest of the oracle plan, using the discovered predicate as needed.
        # For example, if the room is now bright, you can proceed to pick/place actions.

        # Example: Try to pick an object if the room is bright
        # Find an object on the floor
        object_on_floor = None
        for obj_name in positions:
            if "object" in obj_name or "cube" in obj_name or "ball" in obj_name:
                object_on_floor = obj_name
                break

        if object_on_floor is not None:
            try:
                print(f"[Task] Attempting to pick up {object_on_floor} at {light_switch_location}.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_on_floor,
                    location=light_switch_location,
                    approach_distance=0.10,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                print(f"[Task] Picked up {object_on_floor}.")
            except Exception as e:
                print(f"[Task] Failed to pick up {object_on_floor}: {e}")

        # Continue with further plan steps as needed...

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()