# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The plan failed due to a missing predicate. We will systematically try exploration skills
        # to discover which predicate is missing, using only the available skills.

        # Available skills: 
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We'll attempt to use each skill in a safe, exploratory way and observe the environment's response.
        # This will help us infer which predicate is missing or required.

        # For demonstration, we will try to execute each skill with plausible arguments.
        # In a real scenario, you would use the plan and feedback to guide this process.

        # Get a list of objects, drawers, locations, and switches from positions or descriptions
        # For this example, we assume positions contains keys like 'objects', 'drawers', 'locations', 'switches'
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        switches = positions.get('switches', [])

        # Helper: get first available of each type for testing
        test_object = objects[0] if objects else None
        test_drawer = drawers[0] if drawers else None
        test_location = locations[0] if locations else None
        test_switch = switches[0] if switches else None

        # Track which skills succeed/fail for exploration
        exploration_results = {}

        # 1. Try to turn on the light if the room is dark (execute_push_switch)
        try:
            if test_switch and test_location:
                print("[Exploration] Trying execute_push_switch...")
                obs, reward, done = execute_push_switch(env, task, test_switch, test_location)
                exploration_results['execute_push_switch'] = 'success'
                print("[Exploration] execute_push_switch succeeded.")
            else:
                exploration_results['execute_push_switch'] = 'skipped'
        except Exception as e:
            print("[Exploration] execute_push_switch failed:", e)
            exploration_results['execute_push_switch'] = 'fail'

        # 2. Try to pick up an object (execute_pick)
        try:
            if test_object and test_location:
                print("[Exploration] Trying execute_pick...")
                obs, reward, done = execute_pick(env, task, test_object, test_location)
                exploration_results['execute_pick'] = 'success'
                print("[Exploration] execute_pick succeeded.")
            else:
                exploration_results['execute_pick'] = 'skipped'
        except Exception as e:
            print("[Exploration] execute_pick failed:", e)
            exploration_results['execute_pick'] = 'fail'

        # 3. Try to open a drawer (execute_pull)
        try:
            # Need a handle object for the drawer; assume first object is a handle for the first drawer
            if test_drawer and test_object and test_location:
                print("[Exploration] Trying execute_pull...")
                obs, reward, done = execute_pull(env, task, test_drawer, test_object, test_location)
                exploration_results['execute_pull'] = 'success'
                print("[Exploration] execute_pull succeeded.")
            else:
                exploration_results['execute_pull'] = 'skipped'
        except Exception as e:
            print("[Exploration] execute_pull failed:", e)
            exploration_results['execute_pull'] = 'fail'

        # 4. Try to place an object in a drawer (execute_place)
        try:
            if test_object and test_drawer and test_location:
                print("[Exploration] Trying execute_place...")
                obs, reward, done = execute_place(env, task, test_object, test_drawer, test_location)
                exploration_results['execute_place'] = 'success'
                print("[Exploration] execute_place succeeded.")
            else:
                exploration_results['execute_place'] = 'skipped'
        except Exception as e:
            print("[Exploration] execute_place failed:", e)
            exploration_results['execute_place'] = 'fail'

        # 5. Try to push (close) a drawer (execute_push)
        try:
            if test_drawer and test_location:
                print("[Exploration] Trying execute_push...")
                obs, reward, done = execute_push(env, task, test_drawer, test_location)
                exploration_results['execute_push'] = 'success'
                print("[Exploration] execute_push succeeded.")
            else:
                exploration_results['execute_push'] = 'skipped'
        except Exception as e:
            print("[Exploration] execute_push failed:", e)
            exploration_results['execute_push'] = 'fail'

        # 6. Try to move the robot (execute_go)
        try:
            if len(locations) >= 2:
                from_location = locations[0]
                to_location = locations[1]
                print("[Exploration] Trying execute_go...")
                obs, reward, done = execute_go(env, task, from_location, to_location)
                exploration_results['execute_go'] = 'success'
                print("[Exploration] execute_go succeeded.")
            else:
                exploration_results['execute_go'] = 'skipped'
        except Exception as e:
            print("[Exploration] execute_go failed:", e)
            exploration_results['execute_go'] = 'fail'

        # 7. Try to sweep an object (execute_sweep)
        try:
            if test_object and test_location:
                print("[Exploration] Trying execute_sweep...")
                obs, reward, done = execute_sweep(env, task, test_object, test_location)
                exploration_results['execute_sweep'] = 'success'
                print("[Exploration] execute_sweep succeeded.")
            else:
                exploration_results['execute_sweep'] = 'skipped'
        except Exception as e:
            print("[Exploration] execute_sweep failed:", e)
            exploration_results['execute_sweep'] = 'fail'

        # 8. Try to use the gripper (execute_gripper)
        try:
            print("[Exploration] Trying execute_gripper...")
            obs, reward, done = execute_gripper(env, task)
            exploration_results['execute_gripper'] = 'success'
            print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)
            exploration_results['execute_gripper'] = 'fail'

        print("\n[Exploration Results]")
        for skill, result in exploration_results.items():
            print(f"  {skill}: {result}")

        # === End of Exploration Phase ===

        # TODO: After exploration, analyze which skill(s) failed and check logs/errors to infer the missing predicate.
        # You can now use this information to update your domain/problem or inform the user.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()