# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The goal is to explore and discover which predicate is missing, specifically related to the robot's location at the light switch.

        # We assume that the environment contains a location called 'light_switch' or similar.
        # Try to move the robot to the light switch location using available skills.

        # Find the robot's current location and the light switch location
        robot_location = None
        light_switch_location = None

        # Try to infer the robot's current location and the light switch location from positions
        for name, pos in positions.items():
            if 'robot' in name or 'Robot' in name:
                robot_location = name
            if 'light_switch' in name or 'switch' in name or 'LightSwitch' in name:
                light_switch_location = name

        if robot_location is None:
            # Fallback: try to get from observation or default
            robot_location = 'robot'
        if light_switch_location is None:
            # Fallback: try to get from observation or default
            light_switch_location = 'light_switch'

        print(f"[Exploration] Robot location: {robot_location}")
        print(f"[Exploration] Light switch location: {light_switch_location}")

        # Try to move the robot to the light switch using the available skill 'execute_go'
        try:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {light_switch_location} using execute_go.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {light_switch_location}.")
        except Exception as e:
            print(f"[Exploration] Failed to move robot to {light_switch_location}: {e}")

        # After moving, try to push the switch if possible (turn on the light)
        # Use 'execute_push_switch' if available in skill_code
        try:
            print(f"[Exploration] Attempting to push the light switch at {light_switch_location}.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name=light_switch_location,
                location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Light switch pushed at {light_switch_location}.")
        except Exception as e:
            print(f"[Exploration] Failed to push light switch: {e}")

        # At this point, check if the predicate (robot-at light_switch) is now true in the environment
        # This would be part of the observation or state
        # For demonstration, print the current observation/state
        try:
            current_obs = task.get_observation()
            print(f"[Exploration] Current observation after exploration: {current_obs}")
        except Exception as e:
            print(f"[Exploration] Could not retrieve observation: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with the oracle plan or further task steps as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()