# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will attempt to execute each available skill in a safe, exploratory way
        # to see which predicates are required but missing, based on action failures.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will try to execute each skill with plausible arguments.
        # If an action fails due to a missing predicate, we log the error.
        # This helps us discover which predicates are required for successful planning.

        # Example objects and locations (replace with actual keys from positions if needed)
        # We'll try to infer some plausible arguments from the positions dictionary.
        # This is a generic exploration; in a real scenario, you would use actual object names.
        object_names = [k for k in positions.keys() if 'drawer' not in k and 'switch' not in k]
        drawer_names = [k for k in positions.keys() if 'drawer' in k]
        switch_names = [k for k in positions.keys() if 'switch' in k]
        location_names = list(set([k.split('_')[0] for k in positions.keys()]))

        # Fallbacks if not enough info
        if not object_names:
            object_names = ['object_1']
        if not drawer_names:
            drawer_names = ['drawer_1']
        if not switch_names:
            switch_names = ['switch_1']
        if not location_names:
            location_names = ['location_1', 'location_2']

        # Try each skill with dummy/plausible arguments
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # Try to pick the first object at the first location
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj=object_names[0],
                        p=location_names[0]
                    )
                elif skill == 'execute_place':
                    # Try to place the first object into the first drawer at the first location
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj=object_names[0],
                        d=drawer_names[0],
                        p=location_names[0]
                    )
                elif skill == 'execute_push':
                    # Try to push the first drawer at the first location
                    obs, reward, done = execute_push(
                        env,
                        task,
                        d=drawer_names[0],
                        p=location_names[0]
                    )
                elif skill == 'execute_pull':
                    # Try to pull the first drawer using the first object as handle at the first location
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        d=drawer_names[0],
                        h=object_names[0],
                        p=location_names[0]
                    )
                elif skill == 'execute_sweep':
                    # Try to sweep the first object at the first location
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        o=object_names[0],
                        p=location_names[0]
                    )
                elif skill == 'execute_rotate':
                    # If rotate skill exists, try to rotate the first object at the first location
                    obs, reward, done = execute_rotate(
                        env,
                        task,
                        o=object_names[0],
                        p=location_names[0]
                    )
                elif skill == 'execute_go':
                    # Try to move from location_1 to location_2
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_loc=location_names[0],
                        to_loc=location_names[1]
                    )
                elif skill == 'execute_gripper':
                    # Try to execute gripper action
                    obs, reward, done = execute_gripper(
                        env,
                        task
                    )
                print(f"[Exploration] Skill {skill} executed successfully.")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed. Possible missing predicate or precondition: {e}")

        # === End of Exploration Phase ===

        # After exploration, you would analyze the output to determine which predicate is missing.
        # For now, the code simply attempts all skills and reports failures.

        # === (Optional) Main Task Plan Execution ===
        # Here you would execute the actual oracle plan step-by-step using the available skills,
        # now that you have explored and possibly identified missing predicates.
        # For this task, the focus is on the exploration phase.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()