# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (room-bright) ===
        # The feedback indicates that (room-bright) is missing, which is required for most actions.
        # We must ensure the room is bright before proceeding with any further actions.
        # According to the domain, the only way to achieve (room-bright) is to use execute_push_switch.

        # Find the switch and its location
        switch_name = None
        switch_location = None
        for obj_name, pos in positions.items():
            if 'switch' in obj_name.lower():
                switch_name = obj_name
                switch_location = pos
                break

        if switch_name is None or switch_location is None:
            print("[Exploration] No switch found in the environment! Cannot turn on the light.")
            return

        # Find the robot's current location
        robot_location = None
        for obj_name, pos in positions.items():
            if 'robot' in obj_name.lower():
                robot_location = pos
                break

        # If robot location is not found, try to get from observation
        if robot_location is None:
            if hasattr(task, 'get_robot_location'):
                robot_location = task.get_robot_location()
            else:
                # Fallback: use a default or the first location in positions
                for obj_name, pos in positions.items():
                    if 'location' in obj_name.lower() or 'room' in obj_name.lower():
                        robot_location = pos
                        break

        # If robot is not at the switch location, move there
        try:
            if robot_location != switch_location:
                # Use execute_go to move to the switch location
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=switch_location
                )
                print(f"[Exploration] Robot moved to switch at {switch_location}")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Now, execute_push_switch to turn on the light (room-bright)
        try:
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name,
                switch_location
            )
            print(f"[Exploration] Executed push switch: {switch_name} at {switch_location}")
        except Exception as e:
            print(f"[Exploration] Error during execute_push_switch: {e}")
            return

        # === Main Task Plan (Example) ===
        # At this point, the room should be bright and further actions can proceed.
        # Insert your oracle plan here, using only the available skills.
        # For demonstration, we will attempt to pick and place an object if available.

        # Find an object on the floor to pick
        object_to_pick = None
        object_location = None
        for obj_name, pos in positions.items():
            if 'object' in obj_name.lower() or 'ball' in obj_name.lower() or 'cube' in obj_name.lower():
                object_to_pick = obj_name
                object_location = pos
                break

        if object_to_pick is not None and object_location is not None:
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_to_pick,
                    object_location
                )
                print(f"[Task] Picked up object: {object_to_pick} at {object_location}")
            except Exception as e:
                print(f"[Task] Error during execute_pick: {e}")

            # Find a drawer to place the object
            drawer_name = None
            drawer_location = None
            for obj_name, pos in positions.items():
                if 'drawer' in obj_name.lower():
                    drawer_name = obj_name
                    drawer_location = pos
                    break

            if drawer_name is not None and drawer_location is not None:
                try:
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_to_pick,
                        drawer_name,
                        drawer_location
                    )
                    print(f"[Task] Placed object: {object_to_pick} in drawer: {drawer_name} at {drawer_location}")
                except Exception as e:
                    print(f"[Task] Error during execute_place: {e}")
            else:
                print("[Task] No drawer found to place the object.")
        else:
            print("[Task] No suitable object found to pick.")

        # === End of Task Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()