# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try all available skills to find missing predicates ===
        # The goal is to identify which predicate is missing by attempting all skills and observing failures.
        # This is especially important if the planner times out or fails due to missing predicates.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to call each skill with plausible arguments.
        # We will catch exceptions and print out which skills fail, to help identify missing predicates.

        # Get a list of objects, drawers, locations, switches from positions or descriptions
        # This is a generic extraction; adapt as needed for your environment
        objects = []
        drawers = []
        locations = []
        handles = []
        switches = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'handle' in name:
                handles.append(name)
            elif 'switch' in name:
                switches.append(name)
            elif 'loc' in name or 'room' in name or 'location' in name:
                locations.append(name)
            else:
                objects.append(name)
        # Fallback: if locations is empty, use a default
        if not locations:
            locations = ['location_1', 'location_2']

        # Try each skill with dummy arguments, catch and print exceptions
        print("=== [Exploration] Trying all available skills to detect missing predicates ===")
        for skill in available_skills:
            try:
                if skill == 'execute_pick':
                    # Needs (object, location)
                    if objects and locations:
                        print(f"[Exploration] Trying {skill} with ({objects[0]}, {locations[0]})")
                        obs, reward, done = execute_pick(env, task, objects[0], locations[0])
                elif skill == 'execute_place':
                    # Needs (object, drawer, location)
                    if objects and drawers and locations:
                        print(f"[Exploration] Trying {skill} with ({objects[0]}, {drawers[0]}, {locations[0]})")
                        obs, reward, done = execute_place(env, task, objects[0], drawers[0], locations[0])
                elif skill == 'execute_push':
                    # Needs (drawer, location)
                    if drawers and locations:
                        print(f"[Exploration] Trying {skill} with ({drawers[0]}, {locations[0]})")
                        obs, reward, done = execute_push(env, task, drawers[0], locations[0])
                elif skill == 'execute_pull':
                    # Needs (drawer, handle, location)
                    if drawers and handles and locations:
                        print(f"[Exploration] Trying {skill} with ({drawers[0]}, {handles[0]}, {locations[0]})")
                        obs, reward, done = execute_pull(env, task, drawers[0], handles[0], locations[0])
                elif skill == 'execute_sweep':
                    # Needs (object, location)
                    if objects and locations:
                        print(f"[Exploration] Trying {skill} with ({objects[0]}, {locations[0]})")
                        obs, reward, done = execute_sweep(env, task, objects[0], locations[0])
                elif skill == 'execute_rotate':
                    # No parameters (if not implemented, skip)
                    print(f"[Exploration] Trying {skill} (no parameters)")
                    obs, reward, done = execute_rotate(env, task)
                elif skill == 'execute_go':
                    # Needs (from_location, to_location)
                    if len(locations) >= 2:
                        print(f"[Exploration] Trying {skill} with ({locations[0]}, {locations[1]})")
                        obs, reward, done = execute_go(env, task, locations[0], locations[1])
                elif skill == 'execute_gripper':
                    # No parameters
                    print(f"[Exploration] Trying {skill} (no parameters)")
                    obs, reward, done = execute_gripper(env, task)
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed: {e}")

        print("=== [Exploration] Finished trying all skills. Check above for missing predicate clues. ===")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this template, you can insert your plan logic below, using only the provided skills.

        # Example (commented out):
        # obs, reward, done = execute_go(env, task, 'location_1', 'location_2')
        # if done:
        #     print("[Task] Task ended after moving!")
        #     return

        # TODO: Insert your oracle plan execution here, using only the provided skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()