# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The goal is to explore and discover which predicate is missing, specifically related to the robot's location at the light switch.
        # We will use available skills to move the robot and observe the state.

        # Step 1: Identify all locations and the light switch
        # For this example, we assume 'light_switch' is a key in positions
        # If not, handle gracefully
        try:
            light_switch_pos = positions['light_switch']
        except KeyError:
            print("[Exploration] 'light_switch' not found in object positions. Exploration aborted.")
            return

        # Find current robot location
        robot_location = None
        for key, value in positions.items():
            if key.startswith('robot') or key == 'robot':
                robot_location = value
                break
        if robot_location is None:
            print("[Exploration] Robot location not found in positions. Exploration aborted.")
            return

        # Step 2: Try to move the robot to the light switch using execute_go
        # The available skill is 'execute_go'
        # The domain requires (robot-free), (robot-at ?from), (room-bright)
        # If the room is dark, we may need to turn on the light first using execute_push_switch

        # For this exploration, we will:
        # 1. Try to move to the light switch location
        # 2. If preconditions are not met (e.g., room is dark), try to turn on the light
        # 3. After moving, check if the predicate (robot-at light_switch) is now true

        # Step 2.1: Try to move to the light switch
        print("[Exploration] Attempting to move robot to light_switch...")

        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=light_switch_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Robot moved to light_switch.")
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")
            # Step 2.2: If failed, try to turn on the light first
            print("[Exploration] Attempting to turn on the light using execute_push_switch...")
            try:
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch='light_switch',
                    location=robot_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] Light switch pushed. Retrying move...")
                # Retry moving to the light switch
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=light_switch_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] Robot moved to light_switch after turning on the light.")
            except Exception as e2:
                print(f"[Exploration] Failed to turn on light or move: {e2}")
                return

        # Step 3: After moving, check if the predicate (robot-at light_switch) is true
        # This would typically be checked via the environment's state or observation
        # For demonstration, we print the current robot position
        try:
            positions = get_object_positions()
            robot_location_after = None
            for key, value in positions.items():
                if key.startswith('robot') or key == 'robot':
                    robot_location_after = value
                    break
            if robot_location_after == light_switch_pos:
                print("[Exploration] Predicate (robot-at light_switch) is now true.")
            else:
                print("[Exploration] Predicate (robot-at light_switch) is still missing or not updated.")
        except Exception as e:
            print(f"[Exploration] Could not verify robot position: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with further task plan if needed

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()