# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # We need to explore the environment to find the location of the light switch
        # and verify if the predicate (robot-at light_switch) is missing or not.

        # Try to find the light switch in the environment
        light_switch_pos = None
        for obj_name, pos in positions.items():
            if "light_switch" in obj_name or "switch" in obj_name:
                light_switch_pos = pos
                light_switch_name = obj_name
                break

        if light_switch_pos is None:
            print("[Exploration] No light switch found in object positions. Exploration failed.")
            return

        print(f"[Exploration] Light switch found: {light_switch_name} at {light_switch_pos}")

        # Try to move the robot to the light switch location using the available skill
        # The available skill for moving is 'execute_go'
        # We need to determine the robot's current location and the name of the light switch location

        # Find robot's current location
        robot_location = None
        for obj_name, pos in positions.items():
            if "robot" in obj_name or "robby" in obj_name:
                robot_location = obj_name
                break

        # If robot's location is not explicitly given, try to infer from observation
        if robot_location is None:
            # Fallback: try to get from observation descriptions
            for desc in descriptions:
                if "robot-at" in desc:
                    robot_location = desc.split()[-1].replace(")", "")
                    break

        if robot_location is None:
            print("[Exploration] Could not determine robot's current location.")
            return

        print(f"[Exploration] Robot is at: {robot_location}")

        # Now, try to move the robot to the light switch location
        # The skill signature is: execute_go(env, task, from_location, to_location)
        try:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {light_switch_name}...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=light_switch_name
            )
            print(f"[Exploration] Robot moved to {light_switch_name}.")
        except Exception as e:
            print(f"[Exploration] Failed to move robot to light switch: {e}")
            return

        # After moving, check if the predicate (robot-at light_switch) is now true
        # This can be done by checking the observation or the environment state
        # For demonstration, print out the new observation descriptions
        new_descriptions = task.get_observation()
        found_predicate = False
        for desc in new_descriptions:
            if "robot-at" in desc and light_switch_name in desc:
                found_predicate = True
                print(f"[Exploration] Predicate found: {desc}")
                break

        if not found_predicate:
            print(f"[Exploration] Predicate (robot-at {light_switch_name}) is still missing after move.")
        else:
            print(f"[Exploration] Predicate (robot-at {light_switch_name}) is present.")

        # === End of Exploration Phase ===

        # You can now proceed with the rest of the oracle plan, using the available skills,
        # knowing that the missing predicate is (robot-at light_switch).
        # For this exploration task, we stop here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
