# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (hand-empty) is missing or problematic.
        # The domain uses (hand-empty), but the exploration domain uses (handempty) (no dash).
        # We will attempt to execute a pick action and catch any issues related to the missing predicate.

        # Try to find an object on the floor and a location for the robot
        object_on_floor = None
        robot_location = None

        # Try to infer from positions dictionary
        for obj_name, pos_info in positions.items():
            # Assume pos_info contains keys like 'type' and 'state'
            if 'on-floor' in pos_info.get('state', []):
                object_on_floor = obj_name
                break

        # Find robot's current location
        for name, pos_info in positions.items():
            if name.startswith('robot') or pos_info.get('type') == 'robot':
                robot_location = pos_info.get('location')
                break

        # Fallbacks if not found
        if object_on_floor is None:
            # Try to pick any object
            for obj_name, pos_info in positions.items():
                if pos_info.get('type') == 'object':
                    object_on_floor = obj_name
                    break

        if robot_location is None:
            # Try to use a default location
            for name, pos_info in positions.items():
                if pos_info.get('type') == 'location':
                    robot_location = name
                    break

        # If still not found, abort exploration
        if object_on_floor is None or robot_location is None:
            print("[Exploration] Could not find suitable object or robot location for exploration.")
            return

        print(f"[Exploration] Attempting to pick object '{object_on_floor}' at location '{robot_location}' to test for missing predicate.")

        # Try to execute the pick skill and catch errors
        try:
            # The pick skill may require additional parameters; we use only those available
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=object_on_floor,
                location=robot_location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print("[Exploration] Pick action executed successfully. No missing predicate detected in this step.")
        except Exception as e:
            print("[Exploration] Exception occurred during pick action:")
            print(e)
            # Check if the error message mentions 'hand-empty' or similar
            if 'hand-empty' in str(e) or 'handempty' in str(e):
                print("[Exploration] Detected missing predicate: (hand-empty) or (handempty).")
                print("[Exploration] The domain and exploration PDDL use different predicate names for empty hand state.")
                print("[Exploration] Please ensure consistency between (hand-empty) and (handempty) in your domain/problem files.")
            else:
                print("[Exploration] Exception does not directly indicate missing (hand-empty) predicate. Please check logs for more details.")

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
