# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (hand-empty) is missing or problematic.
        # We will attempt to use skills that require (hand-empty) as a precondition,
        # and observe if the environment or skill call fails due to this predicate.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We'll attempt to execute 'execute_pick' and 'execute_place', which require (hand-empty) or (handempty).
        # If an error occurs, we log it and infer the missing predicate.

        # For demonstration, we will try to pick up the first object on the floor, if any.
        # We will also try to place it if possible.

        # Find an object on the floor and a location
        object_on_floor = None
        location = None
        drawer = None
        handle = None

        # Try to extract object and location names from positions dict
        for name, pos in positions.items():
            if 'object' in name or 'obj' in name:
                object_on_floor = name
                break
        for name, pos in positions.items():
            if 'location' in name or 'room' in name or 'loc' in name:
                location = name
                break
        for name, pos in positions.items():
            if 'drawer' in name:
                drawer = name
                break
        for name, pos in positions.items():
            if 'handle' in name:
                handle = name
                break

        # Fallbacks if not found
        if object_on_floor is None:
            object_on_floor = list(positions.keys())[0]
        if location is None:
            location = list(positions.keys())[0]
        if drawer is None:
            drawer = list(positions.keys())[0]
        if handle is None:
            handle = list(positions.keys())[0]

        print(f"[Exploration] Trying to pick object: {object_on_floor} at location: {location}")

        # Try to execute 'execute_pick'
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=object_on_floor,
                location_name=location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print("[Exploration] execute_pick succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")
            print("[Exploration] This may indicate a missing predicate such as (hand-empty) or (handempty).")

        # Try to execute 'execute_place' if pick succeeded
        try:
            print(f"[Exploration] Trying to place object: {object_on_floor} into drawer: {drawer} at location: {location}")
            obs, reward, done = execute_place(
                env,
                task,
                obj_name=object_on_floor,
                drawer_name=drawer,
                location_name=location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print("[Exploration] execute_place succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")
            print("[Exploration] This may indicate a missing predicate such as (hand-empty) or (handempty).")

        # Try to execute 'execute_gripper' which also requires (robot-free) and (room-bright)
        try:
            print("[Exploration] Trying to execute_gripper.")
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=50,
                timeout=5.0
            )
            print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # Try to execute 'execute_sweep' which requires (robot-free) and (room-bright)
        try:
            print(f"[Exploration] Trying to execute_sweep on object: {object_on_floor} at location: {location}")
            obs, reward, done = execute_sweep(
                env,
                task,
                obj_name=object_on_floor,
                location_name=location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print("[Exploration] execute_sweep succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # Try to execute 'execute_go' to another location if possible
        other_location = None
        for name, pos in positions.items():
            if name != location and ('location' in name or 'room' in name or 'loc' in name):
                other_location = name
                break
        if other_location is not None:
            try:
                print(f"[Exploration] Trying to execute_go from {location} to {other_location}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=location,
                    to_location=other_location,
                    max_steps=100,
                    timeout=10.0
                )
                print("[Exploration] execute_go succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # Exploration summary
        print("[Exploration] Exploration phase complete. If any of the above actions failed due to missing (hand-empty) or (handempty), this is the missing predicate.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()