# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # The feedback indicates that the predicate (robot-at light_switch) is missing.
        # We need to explore the environment to discover the location of the light switch
        # and ensure the robot can reach it.

        # Try to find the light switch in the environment
        light_switch_name = None
        for obj_name in positions:
            if "light_switch" in obj_name or "switch" in obj_name or "light" in obj_name:
                light_switch_name = obj_name
                break

        if light_switch_name is None:
            print("[Exploration] Could not find a light switch object in the environment positions.")
            # Optionally, list all available objects for debugging
            print("[Exploration] Available objects:", list(positions.keys()))
            return

        light_switch_pos = positions[light_switch_name]
        print(f"[Exploration] Found light switch: {light_switch_name} at position {light_switch_pos}")

        # Try to find the robot's current location
        robot_location_name = None
        for obj_name in positions:
            if "robot" in obj_name or "base" in obj_name:
                robot_location_name = obj_name
                break

        # If robot location is not explicitly named, try to infer from observation or default
        if robot_location_name is None:
            # Fallback: assume robot starts at a default location
            robot_location_name = "robot_base"
            if robot_location_name not in positions:
                # If not present, just pick the first location as a fallback
                for obj_name in positions:
                    if "location" in obj_name or "room" in obj_name or "base" in obj_name:
                        robot_location_name = obj_name
                        break

        print(f"[Exploration] Robot initial location: {robot_location_name}")

        # === Move robot to the light switch location using execute_go ===
        # The domain requires (robot-free), (robot-at ?from), (room-bright) for execute_go
        # But if the room is dark, we may need to use execute_push_switch to turn on the light

        # First, check if the room is dark or bright
        # We'll try to infer this from the observation or assume dark if not specified
        room_is_bright = False
        if hasattr(task, "get_room_brightness"):
            room_is_bright = task.get_room_brightness()
        else:
            # Try to infer from obs or descriptions
            if "room-bright" in str(descriptions).lower() or "room-bright" in str(obs).lower():
                room_is_bright = True
            elif "room-dark" in str(descriptions).lower() or "room-dark" in str(obs).lower():
                room_is_bright = False
            else:
                # Default to dark if unknown
                room_is_bright = False

        # If the room is dark, we need to move to the light switch and push it
        # But execute_go requires room-bright, so we may need to use a different skill or handle this as a special case
        # In this domain, only execute_push_switch can be used in the dark, but the robot must already be at the switch

        # So, if the robot is not at the light switch, and the room is dark, we cannot use execute_go
        # We need to check if the robot is already at the light switch location
        robot_at_switch = False
        if robot_location_name == light_switch_name:
            robot_at_switch = True
        else:
            # Optionally, check if their positions are close enough
            try:
                robot_pos = positions[robot_location_name]
                dist = np.linalg.norm(np.array(robot_pos) - np.array(light_switch_pos))
                if dist < 0.2:  # Threshold for "at" (tune as needed)
                    robot_at_switch = True
            except Exception as e:
                print("[Exploration] Could not compare robot and switch positions:", e)
                robot_at_switch = False

        if not room_is_bright:
            print("[Exploration] Room is dark. Need to turn on the light.")
            if not robot_at_switch:
                print("[Exploration] Robot is not at the light switch. Cannot move in the dark (execute_go requires room-bright).")
                print("[Exploration] Exploration reveals missing predicate: (robot-at light_switch) is required for execute_push_switch.")
                print("[Exploration] Please ensure the initial state allows the robot to be at the light switch or the plan includes a way to reach it in the dark.")
                # This is the missing predicate as per feedback
                return
            else:
                # Robot is at the light switch, can push the switch
                try:
                    print("[Exploration] Executing execute_push_switch to turn on the light.")
                    obs, reward, done = execute_push_switch(
                        env,
                        task,
                        switch_name=light_switch_name,
                        location_name=light_switch_name,  # Assume switch is at its own location
                        max_steps=100,
                        timeout=10.0
                    )
                    print("[Exploration] Light switched on.")
                    room_is_bright = True
                except Exception as e:
                    print("[Exploration] Failed to execute execute_push_switch:", e)
                    return
        else:
            print("[Exploration] Room is already bright.")

        # After turning on the light, we can now use execute_go to move to other locations
        # For demonstration, let's move the robot to the light switch location if not already there
        if not robot_at_switch:
            try:
                print(f"[Exploration] Moving robot to light switch location: {light_switch_name}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location_name,
                    to_location=light_switch_name,
                    max_steps=100,
                    timeout=10.0
                )
                print("[Exploration] Robot moved to light switch location.")
            except Exception as e:
                print("[Exploration] Failed to execute execute_go:", e)
                return

        # At this point, the missing predicate (robot-at light_switch) should be satisfied
        print("[Exploration] Exploration complete. Predicate (robot-at light_switch) can now be achieved.")

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan as needed, using only predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()