# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The missing predicate is likely related to the robot's location at the light switch.
        # We need to explore the environment to discover this.

        # Try to move the robot to the light switch location using available skills.
        # The skill 'execute_go' is used for moving between locations.
        # The skill 'execute_push_switch' is used to interact with the light switch.

        # First, find the relevant location names from positions or descriptions.
        # We'll try to robustly find the light switch location.
        light_switch_location = None
        for key in positions:
            if 'light_switch' in key or 'switch' in key:
                light_switch_location = key
                break
        if light_switch_location is None:
            # Fallback: try to find in descriptions
            for desc in descriptions:
                if 'light switch' in desc or 'switch' in desc:
                    light_switch_location = desc
                    break
        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in environment info.")
            return

        # Find the robot's current location
        robot_location = None
        for key in positions:
            if 'robot' in key or 'base' in key:
                robot_location = key
                break
        if robot_location is None:
            # Try to infer from observation
            if hasattr(task, 'get_robot_location'):
                robot_location = task.get_robot_location()
            else:
                # Fallback: use a default or first location
                robot_location = list(positions.keys())[0]

        print(f"[Exploration] Robot starts at: {robot_location}")
        print(f"[Exploration] Light switch location: {light_switch_location}")

        # Try to move the robot to the light switch location
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=light_switch_location
            )
            print(f"[Exploration] Robot moved to light switch location: {light_switch_location}")
        except Exception as e:
            print(f"[Exploration] Failed to move to light switch: {e}")
            return

        # Now, try to push the switch to turn on the light (if possible)
        try:
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name=light_switch_location,
                location=light_switch_location
            )
            print(f"[Exploration] Executed push switch at: {light_switch_location}")
        except Exception as e:
            print(f"[Exploration] Failed to push switch: {e}")

        # After exploration, check if the predicate (robot-at light_switch) is now true
        # This can be checked via the environment or observation
        # For demonstration, print the current robot location if possible
        if hasattr(task, 'get_robot_location'):
            current_loc = task.get_robot_location()
            print(f"[Exploration] Robot is now at: {current_loc}")
        else:
            print("[Exploration] Cannot directly query robot location after exploration.")

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan using the discovered predicate and available skills.
        # For example, now that the room is bright, proceed with pick/place actions as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()