# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (robot-free) ===
        # The feedback indicates that the predicate (robot-free) is missing or not being set as expected.
        # We will perform an exploration using only the available skills to check and ensure the robot is in the 'robot-free' state.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use 'execute_gripper' and 'execute_sweep' to see if they affect the robot-free state,
        # and also check the effect of 'execute_pick' and 'execute_place' as per the domain PDDL.

        # For demonstration, we will:
        # 1. Move the robot to a location with an object (if possible)
        # 2. Attempt to pick the object (which should make robot not free)
        # 3. Place the object (which should make robot free again)
        # 4. Use execute_gripper and execute_sweep to see if they affect robot-free
        # 5. Print/log the state after each action to observe the effect on robot-free

        # --- Step 1: Find an object on the floor and its location ---
        object_on_floor = None
        object_location = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False):
                object_on_floor = obj_name
                object_location = obj_info.get('location', None)
                break

        if object_on_floor is None or object_location is None:
            print("[Exploration] No object on floor found for exploration.")
        else:
            print(f"[Exploration] Found object '{object_on_floor}' on floor at location '{object_location}'.")

            # --- Step 2: Move robot to the object's location ---
            try:
                # Assume we can get the robot's current location from positions or obs
                robot_location = None
                for loc_name, loc_info in positions.items():
                    if loc_info.get('robot_at', False):
                        robot_location = loc_name
                        break
                if robot_location is None:
                    print("[Exploration] Could not determine robot's current location.")
                elif robot_location != object_location:
                    print(f"[Exploration] Moving robot from '{robot_location}' to '{object_location}'.")
                    obs, reward, done = execute_go(env, task, robot_location, object_location)
                    robot_location = object_location
                else:
                    print(f"[Exploration] Robot already at '{object_location}'.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_go: {e}")

            # --- Step 3: Try to pick the object (should make robot not free) ---
            try:
                print(f"[Exploration] Attempting to pick '{object_on_floor}' at '{object_location}'.")
                obs, reward, done = execute_pick(env, task, object_on_floor, object_location)
                print("[Exploration] Executed execute_pick.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick: {e}")

            # --- Step 4: Try to place the object in a drawer (if any open/available) ---
            drawer_found = False
            for obj_name, obj_info in positions.items():
                if obj_info.get('type', '') == 'drawer' and obj_info.get('drawer_open', False):
                    drawer_name = obj_name
                    drawer_location = obj_info.get('location', object_location)
                    try:
                        print(f"[Exploration] Attempting to place '{object_on_floor}' in drawer '{drawer_name}' at '{drawer_location}'.")
                        obs, reward, done = execute_place(env, task, object_on_floor, drawer_name, drawer_location)
                        print("[Exploration] Executed execute_place.")
                        drawer_found = True
                        break
                    except Exception as e:
                        print(f"[Exploration] Exception during execute_place: {e}")
            if not drawer_found:
                print("[Exploration] No open drawer found for execute_place.")

            # --- Step 5: Try execute_gripper (should not affect robot-free) ---
            try:
                print("[Exploration] Executing execute_gripper.")
                obs, reward, done = execute_gripper(env, task)
                print("[Exploration] Executed execute_gripper.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_gripper: {e}")

            # --- Step 6: Try execute_sweep (should not affect robot-free) ---
            try:
                print(f"[Exploration] Executing execute_sweep on '{object_on_floor}' at '{object_location}'.")
                obs, reward, done = execute_sweep(env, task, object_on_floor, object_location)
                print("[Exploration] Executed execute_sweep.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_sweep: {e}")

        # --- End of Exploration Phase ---

        # === Main Task Plan Execution ===
        # At this point, you would continue with the oracle plan for the actual task,
        # using only the available skills and the knowledge gained from the exploration phase.
        # For this code, the focus is on the exploration for the missing predicate (robot-free).

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()