# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The missing predicate is likely the location of the light switch.
        # We need to explore the environment to find the light switch and update our knowledge.

        # Try to move the robot to all known locations to find the light switch
        # Assume positions contains all relevant locations, including possible 'light_switch'
        found_light_switch = False
        robot_current_location = None

        # Find initial robot location
        for loc_name, pos in positions.items():
            if 'robot' in loc_name or 'robot' in descriptions.get('objects', []):
                robot_current_location = loc_name
                break
        # If not found, try to get from observation or fallback
        if robot_current_location is None:
            robot_current_location = 'start'  # fallback, may need adjustment

        # List all locations (excluding robot and objects)
        location_names = [name for name in positions.keys() if 'location' in name or 'room' in name or 'switch' in name or 'drawer' in name or 'light' in name]

        # Try to go to each location and check if it's the light switch
        for loc in location_names:
            if loc == robot_current_location:
                continue
            try:
                print(f"[Exploration] Attempting to go to location: {loc}")
                # Use execute_go skill to move to the location (if available)
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                robot_current_location = loc
                # After moving, check if this is the light switch location
                if 'switch' in loc or 'light' in loc:
                    print(f"[Exploration] Found possible light switch at: {loc}")
                    found_light_switch = True
                    light_switch_location = loc
                    break
            except Exception as e:
                print(f"[Exploration] Could not move to {loc}: {e}")
                continue

        if not found_light_switch:
            print("[Exploration] Light switch location not found during exploration.")
            # Optionally, try to sweep or use other skills to find it
            # For now, abort or continue with fallback
            light_switch_location = None

        # === Main Task Plan ===
        # Now, use the found light switch location to execute the plan

        if found_light_switch and light_switch_location is not None:
            try:
                print(f"[Task] Moving to light switch at {light_switch_location} to turn on the light.")
                # Move to light switch if not already there
                if robot_current_location != light_switch_location:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=robot_current_location,
                        to_location=light_switch_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    robot_current_location = light_switch_location

                # Push the switch to turn on the light
                print("[Task] Executing push switch to turn on the light.")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name=light_switch_location,
                    location=light_switch_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                # Now the room should be bright, continue with the rest of the plan as needed

                # Example: pick up an object on the floor (if any)
                for obj_name, obj_pos in positions.items():
                    if 'object' in obj_name or 'ball' in obj_name:
                        print(f"[Task] Attempting to pick up {obj_name} at {robot_current_location}")
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            object_name=obj_name,
                            location=robot_current_location,
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                        break  # Only pick one for demonstration

                # Continue with other actions as required by the oracle plan...

            except Exception as e:
                print(f"[Task] Exception during main plan execution: {e}")
        else:
            print("[Task] Could not find light switch location, skipping main plan.")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()