# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates missing predicate: (on-floor item1)
        # We need to explore to determine if item1 is on the floor.
        # We'll use available skills to explore the environment.

        # For demonstration, let's assume 'item1' is the object of interest.
        # The actual object names and locations should be retrieved from the environment.
        object_name = 'item1'
        robot_location = None
        object_location = None

        # Try to find the robot and object locations from positions
        try:
            # positions is expected to be a dict: {'item1': (x, y, z), ...}
            if object_name in positions:
                object_location = positions[object_name]
            # Find robot location (assuming 'robot' or similar key)
            for key in positions:
                if 'robot' in key:
                    robot_location = positions[key]
                    break
        except Exception as e:
            print(f"[Exploration] Error retrieving positions: {e}")

        # If locations are not found, set to default or skip
        if object_location is None:
            print(f"[Exploration] Could not find position for {object_name}. Skipping exploration.")
        else:
            print(f"[Exploration] {object_name} is at {object_location}")

        # Exploration: Try to use available skills to check if item1 is on the floor
        # Since we cannot define new skills, we use the available ones.
        # We attempt to pick the object; if successful, it was on the floor.

        exploration_success = False
        try:
            # Move robot to the object's location if needed
            # Assume we have a mapping from positions to location names if required
            # For this example, we use a placeholder location name 'floor_area'
            # In a real scenario, you would map object_location to a location name
            location_name = 'floor_area'
            # If robot is not at the location, move it
            # Use execute_go if available
            if 'execute_go' in globals():
                # For demonstration, use placeholder 'start' and 'floor_area'
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='start',
                    to_location=location_name,
                    max_steps=100
                )
                print(f"[Exploration] Robot moved to {location_name}")
            # Try to pick the object
            if 'execute_pick' in globals():
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_name,
                    location=location_name,
                    max_steps=100
                )
                if done:
                    print(f"[Exploration] Successfully picked {object_name} from {location_name}.")
                    exploration_success = True
                else:
                    print(f"[Exploration] Could not pick {object_name} from {location_name}.")
            else:
                print("[Exploration] execute_pick skill not available.")
        except Exception as e:
            print(f"[Exploration] Exception during exploration: {e}")

        # Based on the result, we infer the missing predicate
        if exploration_success:
            print(f"[Exploration] Predicate (on-floor {object_name}) is TRUE.")
        else:
            print(f"[Exploration] Predicate (on-floor {object_name}) is FALSE or could not be determined.")

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan using the available skills.
        # For demonstration, we show a generic sequence using the available skills.

        try:
            # Example: Turn on the light if the room is dark
            if 'execute_push_switch' in globals():
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name='switch1',
                    location='floor_area',
                    max_steps=50
                )
                print("[Task] Turned on the light.")

            # Example: Pick the object (if not already picked)
            if not exploration_success and 'execute_pick' in globals():
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_name,
                    location='floor_area',
                    max_steps=100
                )
                print(f"[Task] Picked {object_name}.")

            # Example: Place the object in a drawer
            if 'execute_place' in globals():
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=object_name,
                    drawer_name='drawer1',
                    location='floor_area',
                    max_steps=100
                )
                print(f"[Task] Placed {object_name} in drawer1.")

            # Example: Close the drawer
            if 'execute_push' in globals():
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name='drawer1',
                    location='floor_area',
                    max_steps=50
                )
                print("[Task] Closed the drawer.")

        except Exception as e:
            print(f"[Task] Exception during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()