# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try available skills to see which predicate is missing.
        # We use only the provided skills: 
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we attempt each skill and catch exceptions or failures.
        # This will help us identify which precondition/predicate is missing.

        # List of skills to try for exploration
        exploration_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Try each skill with plausible arguments from the environment
        # We assume get_object_positions() returns a dict with object names and their positions
        # For drawers, switches, etc., we may need to infer from the environment or use placeholder names

        # Gather object, drawer, location, and switch names from positions or descriptions
        object_names = []
        drawer_names = []
        location_names = []
        switch_names = []

        # Try to extract names from positions or descriptions
        if positions:
            for name in positions.keys():
                if 'drawer' in name:
                    drawer_names.append(name)
                elif 'switch' in name:
                    switch_names.append(name)
                elif 'loc' in name or 'room' in name or 'position' in name:
                    location_names.append(name)
                else:
                    object_names.append(name)

        # Fallback: If not found, use some default names
        if not location_names:
            location_names = ['location1', 'location2']
        if not object_names:
            object_names = ['object1']
        if not drawer_names:
            drawer_names = ['drawer1']
        if not switch_names:
            switch_names = ['switch1']

        # Assume robot starts at location_names[0]
        robot_location = location_names[0]
        target_location = location_names[1] if len(location_names) > 1 else location_names[0]

        # Try each skill and print the result
        for skill in exploration_skills:
            print(f"[Exploration] Attempting skill: {skill}")
            try:
                if skill == 'execute_pick':
                    # Try picking up the first object at the robot's location
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj=object_names[0],
                        p=robot_location
                    )
                elif skill == 'execute_place':
                    # Try placing the first object into the first drawer at the robot's location
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj=object_names[0],
                        d=drawer_names[0],
                        p=robot_location
                    )
                elif skill == 'execute_push':
                    # Try pushing (closing) the first drawer at the robot's location
                    obs, reward, done = execute_push(
                        env,
                        task,
                        d=drawer_names[0],
                        p=robot_location
                    )
                elif skill == 'execute_pull':
                    # Try pulling (opening) the first drawer using the first object as handle at the robot's location
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        d=drawer_names[0],
                        h=object_names[0],
                        p=robot_location
                    )
                elif skill == 'execute_sweep':
                    # Try sweeping the first object at the robot's location
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        o=object_names[0],
                        p=robot_location
                    )
                elif skill == 'execute_rotate':
                    # If rotate skill exists, try rotating the first object at the robot's location
                    obs, reward, done = execute_rotate(
                        env,
                        task,
                        o=object_names[0],
                        p=robot_location
                    )
                elif skill == 'execute_go':
                    # Try moving from robot_location to target_location
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_loc=robot_location,
                        to_loc=target_location
                    )
                elif skill == 'execute_gripper':
                    # Try gripper action (no parameters)
                    obs, reward, done = execute_gripper(
                        env,
                        task
                    )
                else:
                    print(f"[Exploration] Skill {skill} not recognized.")
                    continue

                print(f"[Exploration] Skill {skill} executed successfully.")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed with exception: {e}")

        print("[Exploration] Exploration phase complete. Check above for missing predicate clues.")

        # === End of Exploration Phase ===

        # After exploration, you would analyze the output to determine which predicate is missing
        # and update your domain/problem files accordingly.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
