# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The missing predicate is likely related to the robot's location at the light switch.
        # We need to explore the environment to discover this.

        # Try to move the robot to all known locations and check for the presence of a light switch.
        # We assume positions contains all relevant locations, including 'light_switch' if present.

        # List of candidate locations to explore
        candidate_locations = list(positions.keys())

        # Try to find the 'light_switch' location
        light_switch_location = None
        for loc in candidate_locations:
            if 'light_switch' in loc or 'switch' in loc:
                light_switch_location = loc
                break

        if light_switch_location is None:
            print("[Exploration] No light switch location found in object positions. Exploration failed.")
            return

        print(f"[Exploration] Found light switch location: {light_switch_location}")

        # Now, try to move the robot to the light switch location using the available skill
        # The available skill for moving is 'execute_go'
        # We need to determine the robot's current location
        robot_location = None
        for loc in candidate_locations:
            if 'robot' in loc or 'start' in loc or 'init' in loc:
                robot_location = loc
                break
        # If not found, try to infer from observation
        if robot_location is None:
            # Try to find a location that is not an object or drawer or switch
            for loc in candidate_locations:
                if not any(x in loc for x in ['drawer', 'object', 'switch', 'handle']):
                    robot_location = loc
                    break
        if robot_location is None:
            # Fallback: just pick the first location
            robot_location = candidate_locations[0]

        print(f"[Exploration] Robot initial location: {robot_location}")

        # Move robot to light switch location
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} using execute_go.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {light_switch_location}.")
        except Exception as e:
            print(f"[Exploration] Failed to move robot to light switch: {e}")
            return

        # Now, try to interact with the light switch using execute_push_switch
        try:
            print(f"[Exploration] Attempting to push the light switch at {light_switch_location}.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name=light_switch_location,
                location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Successfully pushed the light switch.")
        except Exception as e:
            print(f"[Exploration] Failed to push the light switch: {e}")
            return

        # After pushing the switch, check if the room is now bright (predicate discovered)
        # This would be reflected in the environment state, but for this skeleton, we just print the result
        print("[Exploration] Exploration phase complete. Predicate (robot-at light_switch) is now satisfied.")

        # === End of Exploration Phase ===

        # At this point, you can proceed with the rest of the oracle plan as needed,
        # using only the available skills and the discovered predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
