# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try available skills to discover missing predicates ===
        # The goal is to identify which predicate is missing by attempting all available skills
        # and observing which preconditions fail or which actions cannot be executed.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Attempt to execute each skill with plausible arguments
        # This will help identify which predicates are missing in the domain/problem
        # and which actions are blocked due to missing information.

        # For demonstration, we will attempt each skill and catch exceptions.
        # We will print out which skills fail and any error messages.

        # Prepare plausible arguments for each skill based on object positions
        # and typical parameter requirements (object, drawer, location, etc.)
        # Since we don't have the actual object names, we will use keys from positions
        # and make reasonable guesses.

        # Get lists of objects, drawers, locations, switches from positions if possible
        object_names = []
        drawer_names = []
        location_names = []
        switch_names = []

        for name, pos in positions.items():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'switch' in name:
                switch_names.append(name)
            elif 'loc' in name or 'room' in name or 'place' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Fallbacks if lists are empty
        if not object_names:
            object_names = ['object_1']
        if not drawer_names:
            drawer_names = ['drawer_1']
        if not location_names:
            location_names = ['location_1']
        if not switch_names:
            switch_names = ['switch_1']

        # Use the first available for each type
        obj = object_names[0]
        drawer = drawer_names[0]
        loc = location_names[0]
        switch = switch_names[0]

        # For execute_go, need two locations
        if len(location_names) < 2:
            location_names = ['location_1', 'location_2']
        from_loc = location_names[0]
        to_loc = location_names[1]

        # Exploration results
        exploration_results = {}

        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # (object, location)
                    obs, reward, done = execute_pick(env, task, obj, loc)
                elif skill == 'execute_place':
                    # (object, drawer, location)
                    obs, reward, done = execute_place(env, task, obj, drawer, loc)
                elif skill == 'execute_push':
                    # (drawer, location)
                    obs, reward, done = execute_push(env, task, drawer, loc)
                elif skill == 'execute_pull':
                    # (drawer, object, location)
                    obs, reward, done = execute_pull(env, task, drawer, obj, loc)
                elif skill == 'execute_sweep':
                    # (object, location)
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                elif skill == 'execute_rotate':
                    # (object, location)
                    obs, reward, done = execute_rotate(env, task, obj, loc)
                elif skill == 'execute_go':
                    # (from_location, to_location)
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                elif skill == 'execute_gripper':
                    # ()
                    obs, reward, done = execute_gripper(env, task)
                else:
                    print(f"[Exploration] Unknown skill: {skill}")
                    continue
                exploration_results[skill] = "Success"
                print(f"[Exploration] Skill {skill} executed successfully.")
            except Exception as e:
                exploration_results[skill] = f"Failed: {str(e)}"
                print(f"[Exploration] Skill {skill} failed with error: {e}")

        print("\n===== Exploration Results =====")
        for skill, result in exploration_results.items():
            print(f"{skill}: {result}")

        # After exploration, you can analyze which skills failed due to missing predicates.
        # This information can be used to update the domain/problem or guide further debugging.

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()