# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to Identify Missing Predicate ===
        # The feedback suggests a missing predicate is blocking planning.
        # We'll attempt to use available skills to explore the environment and infer missing predicates.
        # Since we cannot define new skills, we use only the provided ones.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to execute each skill in a safe, exploratory way to see if any predicate is missing
        # For each object and location, attempt to use skills and catch exceptions
        # This will help us identify which predicates are missing or which actions fail

        # Get all object and location names from positions
        object_names = []
        location_names = []
        for name, pos in positions.items():
            if 'drawer' in name or 'table' in name or 'room' in name or 'location' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Try to execute_go between locations (if at least two exist)
        if len(location_names) >= 2 and 'execute_go' in available_skills:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] execute_go succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # Try to pick each object at each location
        for obj in object_names:
            for loc in location_names:
                if 'execute_pick' in available_skills:
                    try:
                        print(f"[Exploration] Trying execute_pick for {obj} at {loc}")
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            object_name=obj,
                            location_name=loc,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print(f"[Exploration] execute_pick succeeded for {obj} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # Try to open and close drawers if any
        for drawer in [name for name in location_names if 'drawer' in name]:
            # Try to pull (open) the drawer
            if 'execute_pull' in available_skills:
                for handle in object_names:
                    try:
                        print(f"[Exploration] Trying execute_pull on {drawer} with handle {handle}")
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            drawer_name=drawer,
                            handle_name=handle,
                            location_name=drawer,  # Assume drawer's location is itself
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print(f"[Exploration] execute_pull succeeded on {drawer} with handle {handle}")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed on {drawer} with handle {handle}: {e}")

            # Try to push (close) the drawer
            if 'execute_push' in available_skills:
                try:
                    print(f"[Exploration] Trying execute_push on {drawer}")
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name=drawer,
                        location_name=drawer,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] execute_push succeeded on {drawer}")
                except Exception as e:
                    print(f"[Exploration] execute_push failed on {drawer}: {e}")

        # Try to place each object in each drawer
        for obj in object_names:
            for drawer in [name for name in location_names if 'drawer' in name]:
                if 'execute_place' in available_skills:
                    try:
                        print(f"[Exploration] Trying execute_place for {obj} in {drawer}")
                        obs, reward, done = execute_place(
                            env,
                            task,
                            object_name=obj,
                            drawer_name=drawer,
                            location_name=drawer,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print(f"[Exploration] execute_place succeeded for {obj} in {drawer}")
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj} in {drawer}: {e}")

        # Try to sweep each object at each location
        for obj in object_names:
            for loc in location_names:
                if 'execute_sweep' in available_skills:
                    try:
                        print(f"[Exploration] Trying execute_sweep for {obj} at {loc}")
                        obs, reward, done = execute_sweep(
                            env,
                            task,
                            object_name=obj,
                            location_name=loc,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print(f"[Exploration] execute_sweep succeeded for {obj} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

        # Try to use gripper (if available)
        if 'execute_gripper' in available_skills:
            try:
                print(f"[Exploration] Trying execute_gripper")
                obs, reward, done = execute_gripper(
                    env,
                    task,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] execute_gripper succeeded")
            except Exception as e:
                print(f"[Exploration] execute_gripper failed: {e}")

        # Try to rotate (if available)
        if 'execute_rotate' in available_skills:
            for obj in object_names:
                try:
                    print(f"[Exploration] Trying execute_rotate for {obj}")
                    obs, reward, done = execute_rotate(
                        env,
                        task,
                        object_name=obj,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] execute_rotate succeeded for {obj}")
                except Exception as e:
                    print(f"[Exploration] execute_rotate failed for {obj}: {e}")

        print("[Exploration] Exploration phase complete. Check above for any failed actions indicating missing predicates.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you can proceed to execute the oracle plan if available.
        # For this template, exploration is the main focus due to feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()