# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The robot must explore to discover the 'light_switch' location and the relevant predicate.
        # We will use available skills to move the robot to all known locations and check for the light switch.

        # Assume positions is a dict: {object_name: (x, y, z), ...}
        # Try to find a location that could be the light switch
        light_switch_location = None
        for name in positions:
            if 'light' in name or 'switch' in name:
                light_switch_location = name
                break

        if light_switch_location is None:
            # Fallback: try to find a plausible location
            for name in positions:
                if 'wall' in name or 'panel' in name:
                    light_switch_location = name
                    break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in object positions.")
            # Optionally, print all positions for debugging
            print("[Exploration] Known positions:", positions)
            # End exploration
            shutdown_environment(env)
            return

        print(f"[Exploration] Identified light switch location: {light_switch_location}")

        # Now, move the robot to the light switch location using execute_go
        # Find current robot location
        robot_location = None
        for name in positions:
            if 'robot' in name or 'base' in name:
                robot_location = name
                break
        if robot_location is None:
            # Fallback: try to get from observation or default
            robot_location = 'robot_base' if 'robot_base' in positions else list(positions.keys())[0]

        print(f"[Exploration] Robot current location: {robot_location}")

        # Use execute_go to move to the light switch location (if not already there)
        if robot_location != light_switch_location:
            try:
                print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=light_switch_location,
                    max_steps=100
                )
                print(f"[Exploration] Robot moved to {light_switch_location}.")
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")
        else:
            print(f"[Exploration] Robot already at light switch location.")

        # At this point, the robot should be at the light switch.
        # The missing predicate is likely (robot-at light_switch)
        print("[Exploration] Exploration complete. Missing predicate discovered: (robot-at light_switch)")

        # === End of Exploration Phase ===

        # You can now proceed with the main oracle plan using the discovered predicate and available skills.
        # For example, to turn on the light:
        try:
            print(f"[Task] Attempting to push the light switch using execute_push_switch.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch=light_switch_location,
                location=light_switch_location,
                max_steps=100
            )
            print(f"[Task] Light switch pushed. Room should now be bright.")
        except Exception as e:
            print(f"[Task] Error during execute_push_switch: {e}")

        # Continue with further plan steps as needed, using only available skills.
        # For example, pick/place actions, etc.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()