# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will attempt to execute each available skill in a safe, exploratory way
        # to see which predicates are required but missing, based on action failures.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will try to execute each skill with plausible arguments.
        # If an action fails due to a missing predicate, we log the error.
        # This helps us identify which predicate is missing in the domain/problem.

        # Example: Try to pick up an object from the floor if available
        exploration_results = {}
        for skill_name in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                if skill_name == 'execute_pick':
                    # Try to pick the first object on the floor at the robot's location
                    for obj_name, obj_info in positions.items():
                        if obj_info.get('on_floor', False):
                            robot_loc = obj_info.get('location', None)
                            if robot_loc is not None:
                                obs, reward, done = execute_pick(
                                    env, task, obj_name, robot_loc
                                )
                                exploration_results[skill_name] = "Success"
                                print(f"[Exploration] {skill_name} succeeded on {obj_name} at {robot_loc}")
                                break
                    else:
                        exploration_results[skill_name] = "No suitable object found"
                elif skill_name == 'execute_place':
                    # Try to place a held object into a drawer if possible
                    for obj_name, obj_info in positions.items():
                        if obj_info.get('holding', False):
                            for drawer_name, drawer_info in positions.items():
                                if drawer_info.get('type', '') == 'drawer' and drawer_info.get('open', False):
                                    robot_loc = drawer_info.get('location', None)
                                    if robot_loc is not None:
                                        obs, reward, done = execute_place(
                                            env, task, obj_name, drawer_name, robot_loc
                                        )
                                        exploration_results[skill_name] = "Success"
                                        print(f"[Exploration] {skill_name} succeeded: placed {obj_name} in {drawer_name} at {robot_loc}")
                                        break
                            else:
                                continue
                            break
                    else:
                        exploration_results[skill_name] = "No suitable object/drawer found"
                elif skill_name == 'execute_push':
                    # Try to push a drawer closed if open
                    for drawer_name, drawer_info in positions.items():
                        if drawer_info.get('type', '') == 'drawer' and drawer_info.get('open', False):
                            robot_loc = drawer_info.get('location', None)
                            if robot_loc is not None:
                                obs, reward, done = execute_push(
                                    env, task, drawer_name, robot_loc
                                )
                                exploration_results[skill_name] = "Success"
                                print(f"[Exploration] {skill_name} succeeded: pushed {drawer_name} at {robot_loc}")
                                break
                    else:
                        exploration_results[skill_name] = "No open drawer found"
                elif skill_name == 'execute_pull':
                    # Try to pull a drawer open if closed and unlocked, and holding its handle
                    for drawer_name, drawer_info in positions.items():
                        if drawer_info.get('type', '') == 'drawer' and drawer_info.get('closed', False) and not drawer_info.get('locked', False):
                            handle_name = drawer_info.get('handle', None)
                            if handle_name and positions.get(handle_name, {}).get('holding', False):
                                robot_loc = drawer_info.get('location', None)
                                if robot_loc is not None:
                                    obs, reward, done = execute_pull(
                                        env, task, drawer_name, handle_name, robot_loc
                                    )
                                    exploration_results[skill_name] = "Success"
                                    print(f"[Exploration] {skill_name} succeeded: pulled {drawer_name} with {handle_name} at {robot_loc}")
                                    break
                    else:
                        exploration_results[skill_name] = "No suitable drawer/handle found"
                elif skill_name == 'execute_sweep':
                    # Try to sweep an object on the floor at robot's location
                    for obj_name, obj_info in positions.items():
                        if obj_info.get('on_floor', False):
                            robot_loc = obj_info.get('location', None)
                            if robot_loc is not None:
                                obs, reward, done = execute_sweep(
                                    env, task, obj_name, robot_loc
                                )
                                exploration_results[skill_name] = "Success"
                                print(f"[Exploration] {skill_name} succeeded: swept {obj_name} at {robot_loc}")
                                break
                    else:
                        exploration_results[skill_name] = "No suitable object found"
                elif skill_name == 'execute_rotate':
                    # If rotate is defined, try to rotate a switch or object if present
                    # (Assume rotate is not implemented in this domain, skip)
                    exploration_results[skill_name] = "Not implemented in this domain"
                elif skill_name == 'execute_go':
                    # Try to move the robot from its current location to another location
                    robot_locs = [obj_info['location'] for obj_name, obj_info in positions.items() if obj_info.get('robot', False)]
                    all_locs = set(obj_info['location'] for obj_info in positions.values() if 'location' in obj_info)
                    if robot_locs:
                        from_loc = robot_locs[0]
                        for to_loc in all_locs:
                            if to_loc != from_loc:
                                obs, reward, done = execute_go(
                                    env, task, from_loc, to_loc
                                )
                                exploration_results[skill_name] = "Success"
                                print(f"[Exploration] {skill_name} succeeded: moved from {from_loc} to {to_loc}")
                                break
                        else:
                            exploration_results[skill_name] = "No other location found"
                    else:
                        exploration_results[skill_name] = "Robot location unknown"
                elif skill_name == 'execute_gripper':
                    # Try to execute gripper action if robot is free and room is bright
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results[skill_name] = "Success"
                    print(f"[Exploration] {skill_name} succeeded: gripper action executed")
                else:
                    exploration_results[skill_name] = "Skill not recognized"
            except Exception as e:
                exploration_results[skill_name] = f"Failed: {str(e)}"
                print(f"[Exploration] {skill_name} failed with error: {e}")

        print("[Exploration] Results:", exploration_results)
        print("If any skill failed due to a missing predicate, check the error messages above to identify the missing predicate.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # After identifying the missing predicate, you would update your domain/problem files accordingly
        # and then proceed to execute the oracle plan using the available skills as needed.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
