# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback suggests a missing predicate is blocking planning.
        # We'll attempt to use available skills to interact with the environment and observe effects.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try each skill in a safe, exploratory way to see if any predicate is missing
        # We'll log the result of each action attempt
        print("=== Exploration Phase: Attempting available skills to discover missing predicates ===")
        for skill_name in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                # Prepare dummy/default arguments for each skill
                # These should be replaced with actual objects/locations if known
                if skill_name == 'execute_pick':
                    # Find an object on the floor and a location
                    obj = None
                    loc = None
                    for k, v in positions.items():
                        if 'on-floor' in k or 'object' in k:
                            obj = k
                            break
                    for k, v in positions.items():
                        if 'location' in k or 'room' in k or 'drawer' in k:
                            loc = k
                            break
                    if obj is not None and loc is not None:
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"[Exploration] execute_pick({obj}, {loc}) executed.")
                elif skill_name == 'execute_place':
                    # Find an object being held, a drawer, and a location
                    obj = None
                    drawer = None
                    loc = None
                    for k, v in positions.items():
                        if 'object' in k:
                            obj = k
                            break
                    for k, v in positions.items():
                        if 'drawer' in k:
                            drawer = k
                            break
                    for k, v in positions.items():
                        if 'location' in k or 'room' in k:
                            loc = k
                            break
                    if obj is not None and drawer is not None and loc is not None:
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] execute_place({obj}, {drawer}, {loc}) executed.")
                elif skill_name == 'execute_push':
                    # Find a drawer and a location
                    drawer = None
                    loc = None
                    for k, v in positions.items():
                        if 'drawer' in k:
                            drawer = k
                            break
                    for k, v in positions.items():
                        if 'location' in k or 'room' in k:
                            loc = k
                            break
                    if drawer is not None and loc is not None:
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"[Exploration] execute_push({drawer}, {loc}) executed.")
                elif skill_name == 'execute_pull':
                    # Find a drawer, a handle (object), and a location
                    drawer = None
                    handle = None
                    loc = None
                    for k, v in positions.items():
                        if 'drawer' in k:
                            drawer = k
                            break
                    for k, v in positions.items():
                        if 'handle' in k or 'object' in k:
                            handle = k
                            break
                    for k, v in positions.items():
                        if 'location' in k or 'room' in k:
                            loc = k
                            break
                    if drawer is not None and handle is not None and loc is not None:
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] execute_pull({drawer}, {handle}, {loc}) executed.")
                elif skill_name == 'execute_sweep':
                    # Find an object and a location
                    obj = None
                    loc = None
                    for k, v in positions.items():
                        if 'object' in k:
                            obj = k
                            break
                    for k, v in positions.items():
                        if 'location' in k or 'room' in k:
                            loc = k
                            break
                    if obj is not None and loc is not None:
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"[Exploration] execute_sweep({obj}, {loc}) executed.")
                elif skill_name == 'execute_rotate':
                    # If rotate takes no arguments
                    obs, reward, done = execute_rotate(env, task)
                    print(f"[Exploration] execute_rotate() executed.")
                elif skill_name == 'execute_go':
                    # Find two locations
                    locs = []
                    for k, v in positions.items():
                        if 'location' in k or 'room' in k:
                            locs.append(k)
                    if len(locs) >= 2:
                        obs, reward, done = execute_go(env, task, locs[0], locs[1])
                        print(f"[Exploration] execute_go({locs[0]}, {locs[1]}) executed.")
                elif skill_name == 'execute_gripper':
                    # If gripper takes no arguments
                    obs, reward, done = execute_gripper(env, task)
                    print(f"[Exploration] execute_gripper() executed.")
            except Exception as e:
                print(f"[Exploration] Skill {skill_name} failed: {e}")

        print("=== Exploration Phase Complete ===")
        print("If any skill failed due to a missing predicate, check the error messages above for clues.")

        # === Main Task Plan Execution ===
        # TODO: Insert the actual oracle plan execution here, using only the available skills.
        # For now, this is left as a placeholder, as the main focus is on exploration for missing predicates.

        print("=== Main Task Plan Execution Placeholder ===")
        # Example:
        # obs, reward, done = execute_pick(env, task, 'object_1', 'location_1')
        # if done:
        #     print("[Task] Task completed after picking object_1.")
        #     return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()