# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will attempt to use available skills to explore and confirm the robot's location.

        # Try to use execute_go to move the robot and observe the effect
        # Since we don't have the exact object/room names, we will attempt to infer them from positions
        # and the observation/description if possible.

        # Extract possible locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'floor' in name or 'room' in name or 'drawer' in name or 'table' in name:
                location_names.append(name)
        # Fallback: if no locations found, use all keys
        if not location_names:
            location_names = list(positions.keys())

        # Try to get the robot's current location from descriptions or obs
        robot_location = None
        for desc in descriptions:
            if 'robot-at' in desc:
                # e.g., desc: '(robot-at floor)'
                tokens = desc.replace('(','').replace(')','').split()
                if len(tokens) == 2:
                    robot_location = tokens[1]
                    break
        if robot_location is None:
            # Fallback: try to guess
            robot_location = location_names[0] if location_names else None

        # Print exploration info
        print("[Exploration] Robot initial location:", robot_location)
        print("[Exploration] Known locations:", location_names)

        # Try to move the robot to another location using execute_go
        # This will help us confirm the predicate (robot-at <location>)
        target_location = None
        for loc in location_names:
            if loc != robot_location:
                target_location = loc
                break

        if robot_location and target_location:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {target_location} using execute_go.")
            try:
                obs, reward, done, info = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=target_location
                )
                print(f"[Exploration] Robot moved to {target_location}.")
            except Exception as e:
                print(f"[Exploration] Failed to execute_go: {e}")
        else:
            print("[Exploration] Could not determine suitable locations for execute_go.")

        # After exploration, print out the current state to check for (robot-at <location>)
        try:
            current_obs = task.get_observation()
            print("[Exploration] Current observation after move:", current_obs)
        except Exception as e:
            print(f"[Exploration] Could not get observation: {e}")

        # === End of Exploration Phase ===

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan step-by-step using only the available skills.
        # For demonstration, we show how to use the available skills in sequence.
        # Replace the following with the actual oracle plan as needed.

        # Example: Pick up an object from the floor and place it in a drawer
        # Find an object on the floor and a drawer
        object_on_floor = None
        drawer_name = None
        for name in positions:
            if 'object' in name or 'ball' in name or 'cube' in name:
                object_on_floor = name
            if 'drawer' in name:
                drawer_name = name
        # Fallbacks
        if not object_on_floor and positions:
            object_on_floor = list(positions.keys())[0]
        if not drawer_name and positions:
            drawer_name = list(positions.keys())[-1]

        # Assume the robot is at the correct location and the room is bright
        # 1. execute_pick
        if object_on_floor and robot_location:
            print(f"[Task] Attempting to pick {object_on_floor} at {robot_location}")
            try:
                obs, reward, done, info = execute_pick(
                    env,
                    task,
                    object_name=object_on_floor,
                    location_name=robot_location
                )
                print(f"[Task] Picked {object_on_floor}.")
            except Exception as e:
                print(f"[Task] Failed to execute_pick: {e}")

        # 2. execute_place
        if object_on_floor and drawer_name and robot_location:
            print(f"[Task] Attempting to place {object_on_floor} in {drawer_name} at {robot_location}")
            try:
                obs, reward, done, info = execute_place(
                    env,
                    task,
                    object_name=object_on_floor,
                    drawer_name=drawer_name,
                    location_name=robot_location
                )
                print(f"[Task] Placed {object_on_floor} in {drawer_name}.")
            except Exception as e:
                print(f"[Task] Failed to execute_place: {e}")

        # 3. execute_push (close the drawer)
        if drawer_name and robot_location:
            print(f"[Task] Attempting to push (close) {drawer_name} at {robot_location}")
            try:
                obs, reward, done, info = execute_push(
                    env,
                    task,
                    drawer_name=drawer_name,
                    location_name=robot_location
                )
                print(f"[Task] Pushed (closed) {drawer_name}.")
            except Exception as e:
                print(f"[Task] Failed to execute_push: {e}")

        # 4. execute_sweep (if needed)
        # 5. execute_gripper (if needed)
        # 6. execute_pull (if needed)
        # 7. execute_rotate (if needed)

        # You can expand the above with more steps as required by your oracle plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()