# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # The feedback indicates that the predicate (robot-at light_switch) is missing.
        # We need to explore the environment to discover the location of the light switch
        # and ensure the robot can reach it.

        # Try to find the light switch location in the environment
        light_switch_location = None
        for name in positions:
            if "light_switch" in name or "switch" in name:
                light_switch_location = name
                break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in object positions.")
            # Optionally, list all positions for debugging
            print("[Exploration] Available positions:", list(positions.keys()))
            # End exploration if not found
            return

        print(f"[Exploration] Found light switch location: {light_switch_location}")

        # Now, check if the robot is already at the light switch location
        # We'll assume the robot's current location is available in the observation or positions
        robot_location = None
        for name in positions:
            if "robot" in name:
                robot_location = name
                break
        # If not found, try to get from obs or descriptions
        if robot_location is None:
            # Fallback: try to get from obs or descriptions if available
            robot_location = None  # Could be set if obs provides this info

        # For this code, we assume the robot is not at the light switch yet
        # So, we need to move the robot to the light switch location

        # Use the available skill: 'execute_go'
        # The domain PDDL for execute_go:
        # (:action execute_go
        #   :parameters (?from ?to - location)
        #   :precondition (and (robot-free) (robot-at ?from) (room-bright))
        #   :effect (and (robot-at ?to) (not (robot-at ?from)))
        # )

        # For exploration, we may need to ensure the room is bright before moving
        # But to turn on the light, the room may be dark, so we need to handle this
        # According to the domain, execute_go is not possible in the dark
        # So, if the room is dark, we cannot move; we need to find a way to turn on the light

        # Let's check if the room is dark (from obs or descriptions)
        room_is_dark = False
        if "room-dark" in descriptions or "room-dark" in obs:
            room_is_dark = True

        # If the room is dark, we cannot move; but the only way to turn on the light is to be at the switch
        # So, the missing predicate is that the robot is not at the light switch, and cannot move there in the dark
        # This is the key exploration finding

        print("[Exploration] Exploration result: The missing predicate is (robot-at light_switch).")
        print("[Exploration] The robot cannot move to the light switch in the dark, as per domain constraints.")
        print("[Exploration] To enable the robot to turn on the light, the initial state must include (robot-at light_switch).")

        # === End of Exploration Phase ===

        # The rest of the plan would proceed only if the robot is at the light switch
        # For demonstration, we attempt to execute the light switch action if possible

        # Try to execute the 'execute_push_switch' skill if available and preconditions are met
        # The domain PDDL for execute_push_switch:
        # (:action execute_push_switch
        #   :parameters (?s - switch ?p - location)
        #   :precondition (and (room-dark) (robot-free) (robot-at ?p))
        #   :effect (and (room-bright) (not (room-dark)))
        # )

        # Check if the skill is available
        if 'execute_push_switch' in globals():
            try:
                print(f"[Task] Attempting to turn on the light at location: {light_switch_location}")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name=light_switch_location,
                    location_name=light_switch_location
                )
                print("[Task] Light switch action executed.")
            except Exception as e:
                print(f"[Task] Failed to execute push switch: {e}")
        else:
            print("[Task] Skill 'execute_push_switch' not available in skill_code.")

        # The rest of the plan (pick, place, etc.) would follow here, using only predefined skills
        # For this exploration, we stop after discovering the missing predicate

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
