# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # For this phase, we will attempt to use the available skills to interact with objects and drawers,
        # and observe if any expected state change or information is missing.

        # Get all objects and drawers from positions
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # Try to execute each available skill and observe the result
        # This is a generic exploration loop to help identify missing predicates

        # 1. Try to pick up each object from the floor
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to pick up {obj} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    obj_pos
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}.")
                else:
                    print(f"[Exploration] Could not pick up {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick for {obj}: {e}")

        # 2. Try to open each drawer (pull)
        for drawer in drawer_names:
            # Find handle for this drawer if available
            handle_name = None
            for obj in object_names:
                if f"handle_of_{obj}_{drawer}" in positions or f"handle-{drawer}" in obj:
                    handle_name = obj
                    break
            if handle_name is None:
                # Try to guess handle name
                for obj in object_names:
                    if 'handle' in obj and drawer in obj:
                        handle_name = obj
                        break
            if handle_name is None:
                print(f"[Exploration] No handle found for {drawer}, skipping pull.")
                continue
            try:
                drawer_pos = positions[drawer]
                print(f"[Exploration] Attempting to pull {drawer} using handle {handle_name} at {drawer_pos}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle_name,
                    drawer_pos
                )
                if done:
                    print(f"[Exploration] Successfully pulled (opened) {drawer}.")
                else:
                    print(f"[Exploration] Could not pull (open) {drawer}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull for {drawer}: {e}")

        # 3. Try to place each object into each drawer
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    drawer_pos = positions[drawer]
                    print(f"[Exploration] Attempting to place {obj} into {drawer} at {drawer_pos}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        drawer_pos
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into {drawer}.")
                    else:
                        print(f"[Exploration] Could not place {obj} into {drawer}.")
                except Exception as e:
                    print(f"[Exploration] Exception during execute_place for {obj} into {drawer}: {e}")

        # 4. Try to push (close) each drawer
        for drawer in drawer_names:
            try:
                drawer_pos = positions[drawer]
                print(f"[Exploration] Attempting to push (close) {drawer} at {drawer_pos}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer,
                    drawer_pos
                )
                if done:
                    print(f"[Exploration] Successfully pushed (closed) {drawer}.")
                else:
                    print(f"[Exploration] Could not push (close) {drawer}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_push for {drawer}: {e}")

        # 5. Try to move (go) between locations
        if len(location_names) >= 2:
            for i in range(len(location_names)):
                for j in range(len(location_names)):
                    if i == j:
                        continue
                    from_loc = location_names[i]
                    to_loc = location_names[j]
                    try:
                        print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_loc,
                            to_loc
                        )
                        if done:
                            print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
                        else:
                            print(f"[Exploration] Could not move from {from_loc} to {to_loc}.")
                    except Exception as e:
                        print(f"[Exploration] Exception during execute_go from {from_loc} to {to_loc}: {e}")

        # 6. Try to sweep each object
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                print(f"[Exploration] Attempting to sweep {obj} at {obj_pos}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    obj_pos
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}.")
                else:
                    print(f"[Exploration] Could not sweep {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_sweep for {obj}: {e}")

        # 7. Try to use gripper
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
            else:
                print(f"[Exploration] Could not use gripper.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        print("[Exploration] Exploration phase complete. Review the above logs to identify any missing predicates or failed actions that may indicate missing state information in the domain/problem description.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()