# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (robot-at floor) is missing or problematic.
        # We'll attempt to explore the environment to check for missing predicates related to robot location.

        # Try to use the available skills to move the robot and observe the effect.
        # Since only predefined skills can be used, and 'execute_go' is the move action,
        # we will attempt to move the robot between known locations and check for errors or missing predicates.

        # Get all location names from positions (assuming keys like 'floor', 'drawer_area', etc.)
        location_names = [name for name in positions if 'location' in name or 'floor' in name or 'area' in name or 'room' in name]
        if not location_names:
            # Fallback: try to infer from all position keys
            location_names = list(positions.keys())

        # For exploration, try to move the robot to each location using execute_go
        # and observe if any errors or missing predicates are encountered.
        exploration_success = False
        for i in range(len(location_names)):
            for j in range(len(location_names)):
                if i == j:
                    continue
                from_loc = location_names[i]
                to_loc = location_names[j]
                print(f"[Exploration] Attempting to move robot from '{from_loc}' to '{to_loc}' using execute_go.")
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=from_loc,
                        to_location=to_loc,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] Moved from '{from_loc}' to '{to_loc}' successfully.")
                    exploration_success = True
                    # After a successful move, check if the predicate (robot-at to_loc) is now true.
                    # This can be done by checking the observation or task state if available.
                    # For demonstration, print the current robot location if possible.
                    if hasattr(task, 'get_robot_location'):
                        current_loc = task.get_robot_location()
                        print(f"[Exploration] Robot is now at: {current_loc}")
                    break
                except Exception as e:
                    print(f"[Exploration] Failed to move from '{from_loc}' to '{to_loc}': {e}")
            if exploration_success:
                break

        if not exploration_success:
            print("[Exploration] Could not move robot between any locations. Possible missing predicate or misconfiguration.")
            print("[Exploration] Please check if the predicate (robot-at <location>) is properly defined in the initial state.")

        # === End of Exploration Phase ===

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan step-by-step using only the available skills.
        # For demonstration, we show how to use the skills in sequence.
        # Replace the following with the actual oracle plan steps as needed.

        # Example: Pick up an object from the floor and place it in a drawer (if such objects exist)
        # Find an object on the floor and a drawer location
        object_on_floor = None
        drawer_name = None
        for obj_name in positions:
            if 'object' in obj_name or 'ball' in obj_name or 'cube' in obj_name:
                object_on_floor = obj_name
                break
        for loc_name in positions:
            if 'drawer' in loc_name:
                drawer_name = loc_name
                break

        if object_on_floor and drawer_name:
            print(f"[Task] Attempting to pick up '{object_on_floor}' from the floor.")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_on_floor,
                    location_name='floor',
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Picked up '{object_on_floor}'.")
            except Exception as e:
                print(f"[Task] Failed to pick up '{object_on_floor}': {e}")

            print(f"[Task] Attempting to place '{object_on_floor}' into '{drawer_name}'.")
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=object_on_floor,
                    drawer_name=drawer_name,
                    location_name=drawer_name,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Placed '{object_on_floor}' into '{drawer_name}'.")
            except Exception as e:
                print(f"[Task] Failed to place '{object_on_floor}' into '{drawer_name}': {e}")
        else:
            print("[Task] Could not find suitable object on floor or drawer for demonstration.")

        # TODO: Replace the above with the actual oracle plan steps as required.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()