# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will attempt to execute each available skill in a safe, exploratory way
        # to see which predicates are required but missing, based on action failures.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will try to execute each skill with plausible arguments.
        # We will catch exceptions and print out which predicates might be missing.

        # Get some plausible objects, drawers, locations, switches from positions
        # (Assume keys in positions dict are like: 'object_1', 'drawer_1', 'location_1', etc.)
        objects = [k for k in positions if 'object' in k]
        drawers = [k for k in positions if 'drawer' in k]
        locations = [k for k in positions if 'location' in k or 'room' in k]
        switches = [k for k in positions if 'switch' in k]

        # Fallbacks if not found
        if not objects:
            objects = list(positions.keys())
        if not drawers:
            drawers = list(positions.keys())
        if not locations:
            locations = list(positions.keys())
        if not switches:
            switches = list(positions.keys())

        # Try each skill with dummy arguments and catch errors
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # Needs (object, location)
                    obj = objects[0]
                    loc = locations[0]
                    obs, reward, done = execute_pick(env, task, obj, loc)
                elif skill == 'execute_place':
                    # Needs (object, drawer, location)
                    obj = objects[0]
                    drw = drawers[0]
                    loc = locations[0]
                    obs, reward, done = execute_place(env, task, obj, drw, loc)
                elif skill == 'execute_push':
                    # Needs (drawer, location)
                    drw = drawers[0]
                    loc = locations[0]
                    obs, reward, done = execute_push(env, task, drw, loc)
                elif skill == 'execute_pull':
                    # Needs (drawer, handle-object, location)
                    drw = drawers[0]
                    handle = objects[0]
                    loc = locations[0]
                    obs, reward, done = execute_pull(env, task, drw, handle, loc)
                elif skill == 'execute_sweep':
                    # Needs (object, location)
                    obj = objects[0]
                    loc = locations[0]
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                elif skill == 'execute_rotate':
                    # No parameters (if not implemented, skip)
                    try:
                        obs, reward, done = execute_rotate(env, task)
                    except Exception as e:
                        print(f"[Exploration] Skill {skill} not implemented or failed: {e}")
                        continue
                elif skill == 'execute_go':
                    # Needs (from, to)
                    from_loc = locations[0]
                    to_loc = locations[1] if len(locations) > 1 else locations[0]
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                elif skill == 'execute_gripper':
                    # No parameters
                    obs, reward, done = execute_gripper(env, task)
                print(f"[Exploration] Skill {skill} executed successfully.")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed: {e}")
                print(f"[Exploration] Possible missing predicate or precondition for {skill}.")
                # Optionally, log or collect the error for further analysis

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete. Check above for missing predicates or failed preconditions.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
