# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback suggests a missing predicate is blocking planning.
        # We attempt to execute available skills in a systematic way to trigger errors or discover missing state.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try each skill with plausible arguments to see if any precondition fails due to a missing predicate
        # This is a dry-run/exploration phase to help identify what is missing in the domain/problem

        # For demonstration, we will attempt to:
        # - Move the robot (execute_go)
        # - Pick an object (execute_pick)
        # - Place an object (execute_place)
        # - Pull a drawer (execute_pull)
        # - Push a drawer (execute_push)
        # - Sweep (execute_sweep)
        # - Gripper (execute_gripper)
        # - Rotate (execute_rotate) if available

        # We will catch and print any exceptions or errors to help identify missing predicates

        # Get lists of objects, drawers, locations, switches from positions (if available)
        objects = [k for k in positions.keys() if 'object' in k or 'obj' in k]
        drawers = [k for k in positions.keys() if 'drawer' in k]
        handles = [k for k in positions.keys() if 'handle' in k]
        locations = [k for k in positions.keys() if 'location' in k or 'room' in k or 'pos' in k]
        switches = [k for k in positions.keys() if 'switch' in k or 'light' in k]

        # Fallbacks if not found
        if not locations:
            locations = list(positions.keys())

        # Try to get current robot location
        robot_location = None
        for loc in locations:
            if 'robot' in loc:
                robot_location = loc
                break
        if robot_location is None and locations:
            robot_location = locations[0]

        # Try to get a target location different from robot_location
        target_location = None
        for loc in locations:
            if loc != robot_location:
                target_location = loc
                break
        if target_location is None and len(locations) > 1:
            target_location = locations[1]
        elif target_location is None:
            target_location = robot_location

        # Try to get an object and a drawer
        obj = objects[0] if objects else None
        drawer = drawers[0] if drawers else None
        handle = handles[0] if handles else None
        switch = switches[0] if switches else None

        # Exploration: Try each skill and catch errors
        print("=== Exploration Phase: Attempting available skills to find missing predicates ===")
        # 1. Try execute_go
        try:
            if robot_location and target_location and robot_location != target_location:
                print(f"[Exploration] Trying execute_go from {robot_location} to {target_location}")
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=target_location)
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # 2. Try execute_pick
        try:
            if obj and robot_location:
                print(f"[Exploration] Trying execute_pick on {obj} at {robot_location}")
                obs, reward, done = execute_pick(env, task, object_name=obj, location=robot_location)
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try execute_place
        try:
            if obj and drawer and robot_location:
                print(f"[Exploration] Trying execute_place {obj} into {drawer} at {robot_location}")
                obs, reward, done = execute_place(env, task, object_name=obj, drawer_name=drawer, location=robot_location)
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")

        # 4. Try execute_pull
        try:
            if drawer and handle and robot_location:
                print(f"[Exploration] Trying execute_pull on {drawer} with handle {handle} at {robot_location}")
                obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_name=handle, location=robot_location)
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")

        # 5. Try execute_push
        try:
            if drawer and robot_location:
                print(f"[Exploration] Trying execute_push on {drawer} at {robot_location}")
                obs, reward, done = execute_push(env, task, drawer_name=drawer, location=robot_location)
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")

        # 6. Try execute_sweep
        try:
            if obj and robot_location:
                print(f"[Exploration] Trying execute_sweep on {obj} at {robot_location}")
                obs, reward, done = execute_sweep(env, task, object_name=obj, location=robot_location)
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # 7. Try execute_gripper
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # 8. Try execute_rotate if available
        try:
            if 'execute_rotate' in available_skills:
                print(f"[Exploration] Trying execute_rotate")
                obs, reward, done = execute_rotate(env, task)
        except Exception as e:
            print(f"[Exploration] execute_rotate failed: {e}")

        print("=== Exploration Phase Complete ===")
        print("Check above for any failed actions and error messages indicating missing predicates or state.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here using the available skills.
        # For this skeleton, the focus is on the exploration phase to help identify missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()