# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The goal is to discover if the robot can be "at" the light_switch location.
        # We will attempt to move the robot to all known locations and check if "robot-at light_switch" becomes true.

        # Step 1: Get all known locations from positions
        location_names = []
        for name, pos in positions.items():
            if "location" in name or "switch" in name or "drawer" in name or "room" in name or "floor" in name:
                location_names.append(name)
        # If no explicit location names, fallback to all keys
        if not location_names:
            location_names = list(positions.keys())

        # Step 2: Try to move to each location using execute_go
        # We assume the robot starts at some initial location
        # Try to find the robot's current location
        robot_current_location = None
        for name in location_names:
            if "robot" in name or "start" in name:
                robot_current_location = name
                break
        # If not found, just pick the first location as starting point
        if robot_current_location is None and location_names:
            robot_current_location = location_names[0]

        # Step 3: Exploration - try to move to 'light_switch' if it exists
        target_location = None
        for name in location_names:
            if "light_switch" in name or "switch" in name:
                target_location = name
                break

        if target_location is None:
            print("[Exploration] No light_switch location found in positions. Available locations:", location_names)
        else:
            print(f"[Exploration] Attempting to move robot from {robot_current_location} to {target_location} to test (robot-at {target_location}) predicate.")

            try:
                # Use the predefined skill 'execute_go' to move the robot
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=target_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Moved robot to {target_location}.")
            except Exception as e:
                print(f"[Exploration] Failed to move robot to {target_location}: {e}")

            # After moving, check if the predicate (robot-at light_switch) is now true
            # This would be part of the observation or state
            # For demonstration, print the observation or state info
            try:
                current_obs = task.get_observation()
                print("[Exploration] Current observation after moving to light_switch:", current_obs)
            except Exception as e:
                print(f"[Exploration] Could not retrieve observation: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with the oracle plan or further exploration as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
