# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # Feedback indicates we need to discover the predicate (robot-at light_switch)
        # We'll use available skills to explore the environment and check for the light switch location

        # Try to move the robot to all known locations and check if any correspond to the light switch
        # This assumes positions is a dict: {object_name: (x, y, z), ...}
        # We'll look for a location that could be the light switch

        # First, get all location names from positions
        location_names = []
        for name in positions:
            if "switch" in name or "light" in name or "switch" in name.lower():
                location_names.append(name)
        # If not found, just use all positions as fallback
        if not location_names:
            location_names = list(positions.keys())

        # Try to move to each location using execute_go
        # We need to know the robot's current location; assume it's in descriptions or obs
        # Try to extract robot's current location
        robot_location = None
        if isinstance(descriptions, dict):
            robot_location = descriptions.get('robot_location', None)
        if not robot_location:
            # Try to infer from obs or fallback to first location
            robot_location = None
            for k in positions:
                if "robot" in k:
                    robot_location = k
                    break
            if not robot_location and location_names:
                robot_location = location_names[0]

        # Now, try to move to each candidate location and see if we can trigger the light switch
        found_light_switch = False
        for loc in location_names:
            if loc == robot_location:
                continue
            try:
                print(f"[Exploration] Attempting to move robot from {robot_location} to {loc} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {loc}.")
                # After moving, try to execute the light switch action
                try:
                    print(f"[Exploration] Attempting to push switch at {loc} using execute_push_switch.")
                    obs, reward, done = execute_push_switch(
                        env,
                        task,
                        switch_name=loc,
                        location=loc,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] Successfully executed push switch at {loc}.")
                    found_light_switch = True
                    break
                except Exception as e:
                    print(f"[Exploration] Could not execute push switch at {loc}: {e}")
            except Exception as e:
                print(f"[Exploration] Could not move to {loc}: {e}")

        if not found_light_switch:
            print("[Exploration] Could not find or activate the light switch. Exploration failed.")
        else:
            print("[Exploration] Light switch found and activated. Room should now be bright.")

        # === Continue with Oracle Plan (if any) ===
        # At this point, the missing predicate (robot-at light_switch) should be discovered and the room is bright.
        # You can now proceed with the rest of the oracle plan using the available skills.
        # For example, pick up an object, open a drawer, etc.
        # (This section would be filled in with the actual oracle plan steps.)

        # Example (commented out, as actual plan is not provided):
        # obs, reward, done = execute_pick(env, task, object_name, location, ...)
        # obs, reward, done = execute_place(env, task, object_name, drawer_name, location, ...)
        # etc.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()