# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, specifically (robot-at floor).
        # We will attempt to use the available skills to explore and confirm the robot's location.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will use 'execute_go' to move the robot and check if the predicate (robot-at floor) is updated.
        # Since we do not have the exact object and location names, we will attempt to infer them from positions.

        # Attempt to find a location named 'floor' in the positions dictionary
        floor_location = None
        for key in positions:
            if 'floor' in key.lower():
                floor_location = key
                break

        # If not found, just pick any location as a fallback
        if floor_location is None:
            for key in positions:
                if 'location' in key.lower() or 'room' in key.lower():
                    floor_location = key
                    break

        # If still not found, just pick the first key
        if floor_location is None and len(positions) > 0:
            floor_location = list(positions.keys())[0]

        # Now, try to get the robot's current location
        robot_location = None
        # Try to find a key that represents the robot's current location
        # This is a placeholder; in a real environment, you would query the state
        # For now, we assume the robot starts at 'floor' if available
        if floor_location is not None:
            robot_location = floor_location

        # For exploration, try to move the robot to another location (if available)
        target_location = None
        for key in positions:
            if key != floor_location:
                target_location = key
                break

        # If no other location, just use the same location
        if target_location is None:
            target_location = floor_location

        print(f"[Exploration] Robot initial location: {robot_location}")
        print(f"[Exploration] Target location for execute_go: {target_location}")

        # Try to execute the 'execute_go' skill to move the robot
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=target_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Executed execute_go from {robot_location} to {target_location}")
        except Exception as e:
            print(f"[Exploration] Failed to execute_go: {e}")

        # Now, try to move back to 'floor' to see if (robot-at floor) is set
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=target_location,
                to_location=floor_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Executed execute_go from {target_location} to {floor_location}")
        except Exception as e:
            print(f"[Exploration] Failed to execute_go (return): {e}")

        # At this point, the robot should be at 'floor'
        # The missing predicate is likely (robot-at floor)
        print("[Exploration] Exploration phase complete. If the robot is at 'floor', the missing predicate is (robot-at floor).")

        # === End of Exploration ===

        # TODO: Continue with further task logic or plan execution as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()