# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (robot-at light_switch)
        # The robot needs to know the location of the light switch.
        # We will attempt to move the robot to all known locations and check if the predicate (robot-at light_switch) becomes true.

        # Assume positions contains all known locations, including 'light_switch' if present
        robot_current_location = None
        for loc in positions:
            if 'robot' in positions and positions['robot'] == positions[loc]:
                robot_current_location = loc
                break
        if robot_current_location is None:
            # Fallback: try to get from observation or default
            robot_current_location = 'init_location'

        # Try to go to each location and check for the missing predicate
        found_light_switch = False
        for loc in positions:
            if loc == robot_current_location:
                continue
            try:
                # Use the available skill: execute_go
                print(f"[Exploration] Attempting to move robot from {robot_current_location} to {loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                robot_current_location = loc
                # After moving, check if the robot is at the light switch
                # This can be done by checking the observation or state
                # For demonstration, we check if the location is 'light_switch'
                if loc == 'light_switch':
                    print("[Exploration] Robot is at the light switch. Missing predicate found: (robot-at light_switch)")
                    found_light_switch = True
                    break
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

        if not found_light_switch:
            print("[Exploration] Could not find the light switch location during exploration.")

        # === Main Task Plan ===
        # Now, execute the oracle plan using only the available skills.
        # Example: Turn on the light if in the dark, then perform pick/place actions.

        # 1. If the room is dark, turn on the light using execute_push_switch
        try:
            # Assume we can check the room state from obs or task
            room_is_dark = False
            if hasattr(task, 'is_room_dark'):
                room_is_dark = task.is_room_dark()
            elif 'room-dark' in descriptions:
                room_is_dark = True
            if room_is_dark and found_light_switch:
                print("[Task] Room is dark. Turning on the light using execute_push_switch.")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name='light_switch',
                    location=robot_current_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
        except Exception as e:
            print(f"[Task] Failed to execute_push_switch: {e}")

        # 2. Example: Pick up an object on the floor if present
        try:
            for obj in positions:
                if obj.startswith('object') or obj.startswith('item'):
                    print(f"[Task] Attempting to pick up {obj} at {robot_current_location}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=obj,
                        location=robot_current_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    break  # Only pick one object for demonstration
        except Exception as e:
            print(f"[Task] Failed to execute_pick: {e}")

        # 3. Example: Place the object in a drawer if available
        try:
            for drawer in positions:
                if drawer.startswith('drawer'):
                    print(f"[Task] Attempting to place object in {drawer} at {robot_current_location}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_name=obj,
                        drawer_name=drawer,
                        location=robot_current_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    break
        except Exception as e:
            print(f"[Task] Failed to execute_place: {e}")

        # 4. Example: Close the drawer if open
        try:
            for drawer in positions:
                if drawer.startswith('drawer'):
                    print(f"[Task] Attempting to push (close) {drawer} at {robot_current_location}")
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name=drawer,
                        location=robot_current_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    break
        except Exception as e:
            print(f"[Task] Failed to execute_push: {e}")

        print("[Task] Task execution completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()