# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use the available exploration actions to try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We'll attempt to use the available skills to explore the environment and check for missing predicates.

        # Since we do not have the actual oracle plan, we perform a systematic exploration using the available skills.
        # We assume that the environment provides a list of objects, drawers, and locations via positions or descriptions.

        # Extract object, drawer, and location names from positions or descriptions
        object_names = []
        drawer_names = []
        location_names = []

        # Try to extract names from positions dictionary
        if positions:
            for name, pos in positions.items():
                if "drawer" in name:
                    drawer_names.append(name)
                elif "loc" in name or "location" in name or "room" in name:
                    location_names.append(name)
                else:
                    object_names.append(name)
        else:
            # Fallback: try to extract from descriptions if available
            if "objects" in descriptions:
                object_names = descriptions["objects"]
            if "drawers" in descriptions:
                drawer_names = descriptions["drawers"]
            if "locations" in descriptions:
                location_names = descriptions["locations"]

        # If still empty, use some default names for demonstration
        if not object_names:
            object_names = ["object_1", "object_2"]
        if not drawer_names:
            drawer_names = ["drawer_1"]
        if not location_names:
            location_names = ["location_1", "location_2"]

        # Assume robot starts at the first location
        robot_location = location_names[0]

        # 1. Try to move to each location and attempt to identify objects (simulate exploration)
        for loc in location_names:
            if loc != robot_location:
                try:
                    print(f"[Exploration] Moving robot from {robot_location} to {loc} using execute_go.")
                    obs, reward, done = execute_go(env, task, robot_location, loc)
                    robot_location = loc
                except Exception as e:
                    print(f"[Exploration] Error during execute_go: {e}")

            # Try to pick up each object at this location
            for obj in object_names:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at {loc} using execute_pick.")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {loc}.")
                        # Try to place the object in each drawer at this location
                        for drawer in drawer_names:
                            try:
                                print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc} using execute_place.")
                                obs, reward, done = execute_place(env, task, obj, drawer, loc)
                                if done:
                                    print(f"[Exploration] Successfully placed {obj} in {drawer} at {loc}.")
                                    break
                            except Exception as e:
                                print(f"[Exploration] Error during execute_place: {e}")
                        # If holding the object, try to open a drawer with a handle (simulate pull)
                        for drawer in drawer_names:
                            handle_name = f"handle_{drawer}"
                            try:
                                print(f"[Exploration] Attempting to pull {drawer} with {handle_name} at {loc} using execute_pull.")
                                obs, reward, done = execute_pull(env, task, drawer, handle_name, loc)
                                if done:
                                    print(f"[Exploration] Successfully pulled {drawer} with {handle_name} at {loc}.")
                                    break
                            except Exception as e:
                                print(f"[Exploration] Error during execute_pull: {e}")
                        # After manipulation, try to push (close) the drawer
                        for drawer in drawer_names:
                            try:
                                print(f"[Exploration] Attempting to push (close) {drawer} at {loc} using execute_push.")
                                obs, reward, done = execute_push(env, task, drawer, loc)
                                if done:
                                    print(f"[Exploration] Successfully pushed (closed) {drawer} at {loc}.")
                                    break
                            except Exception as e:
                                print(f"[Exploration] Error during execute_push: {e}")
                        # After all, try to sweep the object (if still on floor)
                        try:
                            print(f"[Exploration] Attempting to sweep {obj} at {loc} using execute_sweep.")
                            obs, reward, done = execute_sweep(env, task, obj, loc)
                        except Exception as e:
                            print(f"[Exploration] Error during execute_sweep: {e}")
                except Exception as e:
                    print(f"[Exploration] Error during execute_pick: {e}")

        # 2. Try to use gripper skill (if available)
        try:
            print(f"[Exploration] Attempting to use gripper using execute_gripper.")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Error during execute_gripper: {e}")

        # 3. Exploration complete. At this point, the missing predicate should be discoverable by analyzing which actions failed due to missing preconditions.
        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
