# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at dice1)
        # The predicate (robot-at dice1) is not recognized in the main domain.
        # We need to explore to determine which predicate is missing.
        # According to the exploration domain, we can use exploration skills to discover missing predicates.

        # Since we do not have exploration skills implemented in skill_code,
        # and only the following skills are available:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # We will use 'execute_go' to move the robot and check if the predicate is recognized.

        # Let's attempt to move the robot to 'dice1' location if it exists in positions.
        # This will help us determine if 'robot-at' with a location argument is valid.

        # Find a location named 'dice1' in the positions dictionary
        dice1_location = None
        for name in positions:
            if name.lower() == 'dice1':
                dice1_location = name
                break

        if dice1_location is not None:
            # Try to get the robot's current location
            robot_location = None
            for name in positions:
                if 'robot' in name.lower():
                    robot_location = name
                    break

            if robot_location is not None:
                print(f"[Exploration] Attempting to move robot from {robot_location} to {dice1_location} to test predicate (robot-at dice1)")
                try:
                    # Use the predefined skill 'execute_go'
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=robot_location,
                        to_location=dice1_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] Moved robot to {dice1_location}. Now checking if predicate (robot-at {dice1_location}) is valid.")
                except Exception as e:
                    print(f"[Exploration] Error during execute_go: {e}")
            else:
                print("[Exploration] Could not determine robot's current location from positions.")
        else:
            print("[Exploration] No object/location named 'dice1' found in positions.")

        # After this exploration, you would check the environment's state or logs to see if (robot-at dice1) is now recognized.
        # This helps identify if the missing predicate is due to a type mismatch or missing declaration in the domain.

        # === End of Exploration Phase ===

        # TODO: Continue with the main task plan after resolving the missing predicate issue.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
