# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # The feedback indicates that the predicate (robot-at light_switch) is missing.
        # We need to explore the environment to discover the location of the light switch
        # and ensure the robot can reach it.

        # Try to find the light switch location in the environment
        light_switch_location = None
        for key in positions:
            if 'light_switch' in key or 'switch' in key:
                light_switch_location = key
                break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in object positions.")
            # Optionally, list all available positions for debugging
            print("[Exploration] Available positions:", list(positions.keys()))
            # End exploration if not found
            shutdown_environment(env)
            return

        print(f"[Exploration] Found light switch location: {light_switch_location}")

        # Get current robot location (if available)
        robot_location = None
        for key in positions:
            if 'robot' in key:
                robot_location = key
                break
        if robot_location is None:
            # Fallback: try to get from observation or assume a default
            robot_location = 'robot_start'
            print("[Exploration] Could not find robot location in object positions. Using default:", robot_location)

        # === Move robot to light switch location using available skill ===
        # Use 'execute_go' skill to move the robot (if not already at the switch)
        try:
            if robot_location != light_switch_location:
                print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=light_switch_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {light_switch_location}.")
            else:
                print(f"[Exploration] Robot already at {light_switch_location}.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            shutdown_environment(env)
            return

        # === Confirm the missing predicate (robot-at light_switch) ===
        # At this point, the robot should be at the light switch location.
        # This confirms the missing predicate is (robot-at light_switch).
        print("[Exploration] Confirmed missing predicate: (robot-at light_switch)")

        # === Now, execute the plan to turn on the light using the available skill ===
        # Use 'execute_push_switch' to turn on the light (if available)
        try:
            print(f"[Task] Executing execute_push_switch at {light_switch_location}.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch=light_switch_location,
                location=light_switch_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Task] Light switch pushed. Room should now be bright.")
        except Exception as e:
            print(f"[Task] Error during execute_push_switch: {e}")
            shutdown_environment(env)
            return

        # === Continue with further plan steps as needed ===
        # For this exploration, we stop after confirming the missing predicate and turning on the light.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
